"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseLocationFilter = exports.parseArrayFilters = exports.isMonitorsQueryFiltered = exports.getSavedObjectKqlFilter = exports.getMonitorFilters = exports.findLocationItem = exports.QuerySchema = exports.OverviewStatusSchema = exports.MONITOR_SEARCH_FIELDS = void 0;
exports.parseMappingKey = parseMappingKey;
exports.validateRouteSpaceName = void 0;
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../../common/constants");
var _sort_field = require("../../common/runtime_types/monitor_management/sort_field");
var _get_all_locations = require("../synthetics_service/get_all_locations");
var _saved_objects = require("../../common/types/saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const StringOrArraySchema = _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())]));
const UseLogicalAndFieldLiterals = _constants.useLogicalAndFields.map(f => _configSchema.schema.literal(f));
const CommonQuerySchema = {
  query: _configSchema.schema.maybe(_configSchema.schema.string()),
  filter: _configSchema.schema.maybe(_configSchema.schema.string()),
  tags: StringOrArraySchema,
  monitorTypes: StringOrArraySchema,
  locations: StringOrArraySchema,
  projects: StringOrArraySchema,
  schedules: StringOrArraySchema,
  status: StringOrArraySchema,
  monitorQueryIds: StringOrArraySchema,
  showFromAllSpaces: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  useLogicalAndFor: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.oneOf(UseLogicalAndFieldLiterals))]))
};
const QuerySchema = exports.QuerySchema = _configSchema.schema.object({
  ...CommonQuerySchema,
  page: _configSchema.schema.maybe(_configSchema.schema.number()),
  perPage: _configSchema.schema.maybe(_configSchema.schema.number()),
  sortField: _sort_field.MonitorSortFieldSchema,
  sortOrder: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('desc'), _configSchema.schema.literal('asc')])),
  searchAfter: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
  internal: _configSchema.schema.maybe(_configSchema.schema.boolean({
    defaultValue: false
  }))
});
const OverviewStatusSchema = exports.OverviewStatusSchema = _configSchema.schema.object({
  ...CommonQuerySchema,
  scopeStatusByLocation: _configSchema.schema.maybe(_configSchema.schema.boolean())
});
const MONITOR_SEARCH_FIELDS = exports.MONITOR_SEARCH_FIELDS = ['name', 'tags.text', 'locations.id.text', 'locations.label', 'urls', 'hosts', 'project_id.text'];
const getMonitorFilters = async (context, attr = _saved_objects.syntheticsMonitorAttributes) => {
  const {
    tags,
    monitorTypes,
    filter = '',
    projects,
    schedules,
    monitorQueryIds,
    locations: queryLocations,
    useLogicalAndFor
  } = context.request.query;
  const locations = await parseLocationFilter(context, queryLocations);
  return parseArrayFilters({
    filter,
    tags,
    monitorTypes,
    projects,
    schedules,
    monitorQueryIds,
    locations
  }, useLogicalAndFor, attr);
};
exports.getMonitorFilters = getMonitorFilters;
const parseArrayFilters = ({
  tags,
  filter,
  configIds,
  projects,
  monitorTypes,
  schedules,
  monitorQueryIds,
  locations
}, useLogicalAndFor = [], attributes = _saved_objects.syntheticsMonitorAttributes) => {
  const filtersStr = [filter, getSavedObjectKqlFilter({
    field: 'tags',
    values: tags,
    operator: useLogicalAndFor.includes('tags') ? 'AND' : 'OR',
    attributes
  }), getSavedObjectKqlFilter({
    field: 'project_id',
    values: projects,
    attributes
  }), getSavedObjectKqlFilter({
    field: 'type',
    values: monitorTypes,
    attributes
  }), getSavedObjectKqlFilter({
    field: 'locations.id',
    values: locations,
    operator: useLogicalAndFor.includes('locations') ? 'AND' : 'OR',
    attributes
  }), getSavedObjectKqlFilter({
    field: 'schedule.number',
    values: schedules,
    attributes
  }), getSavedObjectKqlFilter({
    field: 'id',
    values: monitorQueryIds,
    attributes
  }), getSavedObjectKqlFilter({
    field: 'config_id',
    values: configIds,
    attributes
  })].filter(f => !!f).join(' AND ');
  return {
    filtersStr,
    locationIds: locations
  };
};
exports.parseArrayFilters = parseArrayFilters;
const getSavedObjectKqlFilter = ({
  field,
  values,
  operator = 'OR',
  searchAtRoot = false,
  attributes = _saved_objects.syntheticsMonitorAttributes
}) => {
  if (values === 'All' || Array.isArray(values) && values !== null && values !== void 0 && values.includes('All')) {
    return undefined;
  }
  if ((0, _lodash.isEmpty)(values) || !values) {
    return '';
  }
  let fieldKey = '';
  if (searchAtRoot) {
    fieldKey = `${field}`;
  } else {
    fieldKey = `${attributes}.${field}`;
  }
  if (Array.isArray(values)) {
    return `${fieldKey}:(${values.map(value => `"${(0, _esQuery.escapeQuotes)(value)}"`).join(` ${operator} `)})`;
  }
  return `${fieldKey}:"${(0, _esQuery.escapeQuotes)(values)}"`;
};
exports.getSavedObjectKqlFilter = getSavedObjectKqlFilter;
const parseLocationFilter = async ({
  syntheticsMonitorClient,
  savedObjectsClient,
  server
}, locations) => {
  var _findLocationItem$id2, _findLocationItem2;
  if (!locations || (locations === null || locations === void 0 ? void 0 : locations.length) === 0) {
    return;
  }
  const {
    allLocations
  } = await (0, _get_all_locations.getAllLocations)({
    syntheticsMonitorClient,
    savedObjectsClient,
    server,
    excludeAgentPolicies: true
  });
  if (Array.isArray(locations)) {
    return locations.map(loc => {
      var _findLocationItem$id, _findLocationItem;
      return (_findLocationItem$id = (_findLocationItem = findLocationItem(loc, allLocations)) === null || _findLocationItem === void 0 ? void 0 : _findLocationItem.id) !== null && _findLocationItem$id !== void 0 ? _findLocationItem$id : '';
    }).filter(val => !!val);
  }
  return [(_findLocationItem$id2 = (_findLocationItem2 = findLocationItem(locations, allLocations)) === null || _findLocationItem2 === void 0 ? void 0 : _findLocationItem2.id) !== null && _findLocationItem$id2 !== void 0 ? _findLocationItem$id2 : ''];
};
exports.parseLocationFilter = parseLocationFilter;
const findLocationItem = (query, locations) => {
  return locations.find(({
    id,
    label
  }) => query === id || label === query);
};

/**
 * Returns whether the query is likely to return a subset of monitor objects.
 * Useful where `absoluteTotal` needs to be determined with a separate call
 * @param monitorQuery { MonitorsQuery }
 */
exports.findLocationItem = findLocationItem;
const isMonitorsQueryFiltered = monitorQuery => {
  const {
    query,
    tags,
    monitorTypes,
    locations,
    status,
    filter,
    projects,
    schedules,
    monitorQueryIds
  } = monitorQuery;
  return !!query || !!filter || !!(locations !== null && locations !== void 0 && locations.length) || !!(monitorTypes !== null && monitorTypes !== void 0 && monitorTypes.length) || !!(tags !== null && tags !== void 0 && tags.length) || !!(status !== null && status !== void 0 && status.length) || !!(projects !== null && projects !== void 0 && projects.length) || !!(schedules !== null && schedules !== void 0 && schedules.length) || !!(monitorQueryIds !== null && monitorQueryIds !== void 0 && monitorQueryIds.length);
};
exports.isMonitorsQueryFiltered = isMonitorsQueryFiltered;
function parseMappingKey(key) {
  switch (key) {
    case 'schedule.keyword':
      return 'schedule.number';
    case 'project_id.keyword':
      return 'project_id';
    default:
      return key;
  }
}
const validateRouteSpaceName = async routeContext => {
  const {
    spaceId,
    server,
    request,
    response
  } = routeContext;
  if (spaceId === _common.DEFAULT_SPACE_ID) {
    // default space is always valid
    return {
      spaceId: _common.DEFAULT_SPACE_ID
    };
  }
  try {
    var _server$spaces;
    await ((_server$spaces = server.spaces) === null || _server$spaces === void 0 ? void 0 : _server$spaces.spacesService.getActiveSpace(request));
  } catch (error) {
    var _error$output;
    if (((_error$output = error.output) === null || _error$output === void 0 ? void 0 : _error$output.statusCode) === 404) {
      return {
        spaceId,
        invalidResponse: response.notFound({
          body: {
            message: `Kibana space '${spaceId}' does not exist`
          }
        })
      };
    }
  }
  return {
    invalidResponse: undefined
  };
};
exports.validateRouteSpaceName = validateRouteSpaceName;