"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchPingsParams = void 0;
var _lib = require("../../../lib");
var _client_defaults = require("../../../../common/constants/client_defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSearchPingsParams = ({
  idsToQuery,
  idSize,
  monitorLocationIds,
  range,
  numberOfChecks,
  includeRetests
}) => {
  // Filters for pings
  const queryFilters = [...(includeRetests ? [_client_defaults.SUMMARY_FILTER] : [_client_defaults.FINAL_SUMMARY_FILTER]), (0, _client_defaults.getRangeFilter)({
    from: range.from,
    to: range.to
  }), {
    terms: {
      'monitor.id': idsToQuery
    }
  }];

  // For each ping we want to get the monitor id as an aggregation and the location as a sub-aggregation
  // The location aggregation will have a filter for down checks and a top hits aggregation to get the latest ping

  // Total checks per location per monitor
  const totalChecks = {
    size: numberOfChecks,
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }],
    _source: {
      includes: ['@timestamp', 'summary', 'monitor', 'observer', 'config_id', 'error', 'agent', 'url', 'state', 'tags', 'service', 'labels']
    }
  };

  // Down checks per location per monitor
  const downChecks = {
    range: {
      'summary.down': {
        gte: '1'
      }
    }
  };
  const locationAggs = {
    downChecks: {
      filter: downChecks
    },
    totalChecks: {
      top_hits: totalChecks
    }
  };
  const idAggs = {
    location: {
      terms: {
        field: 'observer.name',
        size: monitorLocationIds.length || 100
      },
      aggs: locationAggs
    }
  };
  const pingAggs = {
    id: {
      terms: {
        field: 'monitor.id',
        size: idSize
      },
      aggs: idAggs
    }
  };
  const params = (0, _lib.createEsParams)({
    body: {
      size: 0,
      query: {
        bool: {
          filter: queryFilters
        }
      },
      aggs: pingAggs
    }
  });
  if (monitorLocationIds.length > 0) {
    params.body.query.bool.filter.push({
      terms: {
        'observer.name': monitorLocationIds
      }
    });
  }
  return params;
};
exports.getSearchPingsParams = getSearchPingsParams;