"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StreamsClient = void 0;
var _elasticsearch = require("@elastic/elasticsearch");
var _esErrors = require("@kbn/es-errors");
var _streamsSchema = require("@kbn/streams-schema");
var _fields = require("./assets/fields");
var _definition_not_found_error = require("./errors/definition_not_found_error");
var _security_error = require("./errors/security_error");
var _status_error = require("./errors/status_error");
var _root_stream_definition = require("./root_stream_definition");
var _state = require("./state_management/state");
var _stream_crud = require("./stream_crud");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * When calling into Elasticsearch, the stack trace is lost.
 * If we create an error before calling, and append it to
 * any stack of the caught error, we get a more useful stack
 * trace.
 */
function wrapEsCall(p) {
  const error = new Error();
  return p.catch(caughtError => {
    caughtError.stack += error.stack;
    throw caughtError;
  });
}
class StreamsClient {
  constructor(dependencies) {
    this.dependencies = dependencies;
  }

  /**
   * Streams is considered enabled when:
   * - the logs root stream exists
   * - it is a wired stream (as opposed to an ingest stream)
   */
  async isStreamsEnabled() {
    const rootLogsStreamExists = await this.getStream(_root_stream_definition.LOGS_ROOT_STREAM_NAME).then(definition => _streamsSchema.Streams.WiredStream.Definition.is(definition)).catch(error => {
      if ((0, _definition_not_found_error.isDefinitionNotFoundError)(error)) {
        return false;
      }
      throw error;
    });
    return rootLogsStreamExists;
  }

  /**
   * Enabling streams means creating the logs root stream.
   * If it is already enabled, it is a noop.
   */
  async enableStreams() {
    const isEnabled = await this.isStreamsEnabled();
    if (isEnabled) {
      return {
        acknowledged: true,
        result: 'noop'
      };
    }
    const result = await _state.State.attemptChanges([{
      type: 'upsert',
      definition: _root_stream_definition.rootStreamDefinition
    }], {
      ...this.dependencies,
      streamsClient: this
    });
    if (result.status === 'failed_with_rollback') {
      throw result.error;
    }
    return {
      acknowledged: true,
      result: 'created'
    };
  }

  /**
   * Disabling streams means deleting the logs root stream
   * AND its descendants, including any Elasticsearch objects,
   * such as data streams. That means it deletes all data
   * belonging to wired streams.
   *
   * It does NOT delete unwired streams.
   */
  async disableStreams() {
    const isEnabled = await this.isStreamsEnabled();
    if (!isEnabled) {
      return {
        acknowledged: true,
        result: 'noop'
      };
    }
    const result = await _state.State.attemptChanges([{
      type: 'delete',
      name: _root_stream_definition.rootStreamDefinition.name
    }], {
      ...this.dependencies,
      streamsClient: this
    });
    if (result.status === 'failed_with_rollback') {
      throw result.error;
    }
    const {
      assetClient,
      storageClient
    } = this.dependencies;
    await Promise.all([assetClient.clean(), storageClient.clean()]);
    return {
      acknowledged: true,
      result: 'deleted'
    };
  }

  /**
   * Resyncing streams means re-installing all Elasticsearch
   * objects (index and component templates, pipelines, and
   * assets), using the stream definitions as the source of
   * truth.
   *
   * Streams are re-synced in a specific order:
   * the leaf nodes are synced first, then its parents, etc.
   * This prevents us from routing to data streams that do
   * not exist yet.
   */
  async resyncStreams() {
    await _state.State.resync({
      ...this.dependencies,
      streamsClient: this
    });
    return {
      acknowledged: true,
      result: 'updated'
    };
  }

  /**
   * Creates or updates a stream. The routing of the parent is
   * also updated (including syncing to Elasticsearch).
   */
  async upsertStream({
    name,
    request
  }) {
    const stream = {
      ...request.stream,
      name
    };
    const result = await _state.State.attemptChanges([{
      type: 'upsert',
      definition: stream
    }], {
      ...this.dependencies,
      streamsClient: this
    });
    if (result.status === 'failed_with_rollback') {
      throw result.error;
    }
    const {
      dashboards,
      queries
    } = request;

    // sync dashboards as before
    await this.dependencies.assetClient.syncAssetList(stream.name, dashboards.map(dashboard => ({
      [_fields.ASSET_ID]: dashboard,
      [_fields.ASSET_TYPE]: 'dashboard'
    })), 'dashboard');

    // sync rules with asset links
    await this.dependencies.queryClient.syncQueries(stream.name, queries);
    return {
      acknowledged: true,
      result: result.changes.created.includes(name) ? 'created' : 'updated'
    };
  }

  /**
   * Forks a stream into a child with a specific condition.
   */
  async forkStream({
    parent,
    name,
    if: condition
  }) {
    const parentDefinition = _streamsSchema.Streams.WiredStream.Definition.parse(await this.getStream(parent));
    const childExistsAlready = await this.existsStream(name);
    if (childExistsAlready) {
      throw new _status_error.StatusError(`Child stream ${name} already exists`, 409);
    }
    const result = await _state.State.attemptChanges([{
      type: 'upsert',
      definition: {
        ...parentDefinition,
        ingest: {
          ...parentDefinition.ingest,
          wired: {
            ...parentDefinition.ingest.wired,
            routing: parentDefinition.ingest.wired.routing.concat({
              destination: name,
              if: condition
            })
          }
        }
      }
    }, {
      type: 'upsert',
      definition: {
        name,
        description: '',
        ingest: {
          lifecycle: {
            inherit: {}
          },
          processing: [],
          wired: {
            fields: {},
            routing: []
          }
        }
      }
    }], {
      ...this.dependencies,
      streamsClient: this
    });
    if (result.status === 'failed_with_rollback') {
      throw result.error;
    }
    return {
      acknowledged: true,
      result: 'created'
    };
  }

  /**
   * Make sure there is a stream definition for a given stream.
   * If the data stream exists but the stream definition does not, it creates an empty stream definition.
   * If the stream definition exists, it is a noop.
   * If the data stream does not exist or the user does not have access, it throws.
   */
  async ensureStream(name) {
    const [streamDefinition, dataStream] = await Promise.all([this.getStoredStreamDefinition(name).catch(error => {
      if ((0, _esErrors.isNotFoundError)(error)) {
        return error;
      }
      throw error;
    }), this.getDataStream(name).catch(error => {
      if ((0, _esErrors.isNotFoundError)(error)) {
        return error;
      }
      throw error;
    })]);
    if (!(0, _esErrors.isNotFoundError)(streamDefinition)) {
      // stream definitely exists, all good
      return;
    }
    if (!(0, _esErrors.isNotFoundError)(dataStream) && (0, _esErrors.isNotFoundError)(streamDefinition)) {
      // stream definition does not exist, but data stream does - create an empty stream definition
      await this.updateStoredStream(this.getDataStreamAsIngestStream(dataStream));
      return;
    }
    // if both do not exist, the stream does not exist, so this should be a 404
    throw streamDefinition;
  }
  getStreamDefinitionFromSource(source) {
    if (!source) {
      throw new _definition_not_found_error.DefinitionNotFoundError(`Cannot find stream definition`);
    }
    return source;
  }

  /**
   * Returns a stream definition for the given name:
   * - if a wired stream definition exists
   * - if an ingest stream definition exists
   * - if a data stream exists (creates an ingest definition on the fly)
   * - if a group stream definition exists
   *
   * Throws when:
   * - no definition is found
   * - the user does not have access to the stream
   */
  async getStream(name) {
    try {
      const response = await this.dependencies.storageClient.get({
        id: name
      });
      const streamDefinition = this.getStreamDefinitionFromSource(response._source);
      if (_streamsSchema.Streams.ingest.all.Definition.is(streamDefinition)) {
        const privileges = await (0, _stream_crud.checkAccess)({
          name,
          scopedClusterClient: this.dependencies.scopedClusterClient
        });
        if (!privileges.read) {
          throw new _security_error.SecurityError(`Cannot read stream, insufficient privileges`);
        }
      }
      return streamDefinition;
    } catch (error) {
      try {
        if ((0, _esErrors.isNotFoundError)(error)) {
          const dataStream = await this.getDataStream(name);
          return this.getDataStreamAsIngestStream(dataStream);
        }
        throw error;
      } catch (e) {
        if ((0, _esErrors.isNotFoundError)(e)) {
          throw new _definition_not_found_error.DefinitionNotFoundError(`Cannot find stream ${name}`);
        }
        throw e;
      }
    }
  }
  async getStoredStreamDefinition(name) {
    return await Promise.all([this.dependencies.storageClient.get({
      id: name
    }).then(response => {
      return this.getStreamDefinitionFromSource(response._source);
    }), (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient: this.dependencies.scopedClusterClient
    }).then(privileges => {
      if (!privileges.read) {
        throw new _security_error.SecurityError(`Cannot read stream, insufficient privileges`);
      }
    })]).then(([wiredDefinition]) => {
      return wiredDefinition;
    });
  }
  async getDataStream(name) {
    return wrapEsCall(this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
      name
    })).then(response => {
      if (response.data_streams.length === 0) {
        throw new _elasticsearch.errors.ResponseError({
          meta: {
            aborted: false,
            attempts: 1,
            connection: null,
            context: null,
            name: 'resource_not_found_exception',
            request: {}
          },
          warnings: [],
          body: 'resource_not_found_exception',
          statusCode: 404
        });
      }
      const dataStream = response.data_streams[0];
      return dataStream;
    });
  }

  /**
   * Checks whether the user has the required privileges to manage the stream.
   * Managing a stream means updating the stream properties. It does not
   * include the dashboard links.
   */
  async getPrivileges(name) {
    const REQUIRED_MANAGE_PRIVILEGES = ['manage_index_templates', 'manage_ingest_pipelines', 'manage_pipeline', 'read_pipeline'];
    const privileges = await this.dependencies.scopedClusterClient.asCurrentUser.security.hasPrivileges({
      cluster: [...REQUIRED_MANAGE_PRIVILEGES, 'monitor_text_structure'],
      index: [{
        names: [name],
        privileges: ['read', 'write', 'create', 'manage', 'monitor', 'manage_data_stream_lifecycle', 'manage_ilm']
      }]
    });
    return {
      manage: REQUIRED_MANAGE_PRIVILEGES.every(privilege => privileges.cluster[privilege] === true) && Object.values(privileges.index[name]).every(privilege => privilege === true),
      monitor: privileges.index[name].monitor,
      lifecycle: privileges.index[name].manage_data_stream_lifecycle && privileges.index[name].manage_ilm,
      simulate: privileges.cluster.read_pipeline && privileges.index[name].create,
      text_structure: privileges.cluster.monitor_text_structure
    };
  }

  /**
   * Creates an on-the-fly ingest stream definition
   * from a concrete data stream.
   */
  getDataStreamAsIngestStream(dataStream) {
    const definition = {
      name: dataStream.name,
      description: '',
      ingest: {
        lifecycle: {
          inherit: {}
        },
        processing: [],
        unwired: {}
      }
    };
    return definition;
  }

  /**
   * Checks whether the stream exists (and whether the
   * user has access to it).
   */
  async existsStream(name) {
    const exists = await this.getStream(name).then(() => true).catch(error => {
      if ((0, _definition_not_found_error.isDefinitionNotFoundError)(error)) {
        return false;
      }
      throw error;
    });
    return exists;
  }

  /**
   * Lists both managed and unmanaged streams
   */
  async listStreams() {
    const streams = await this.listStreamsWithDataStreamExistence();
    return streams.map(({
      stream
    }) => {
      return stream;
    });
  }
  async listStreamsWithDataStreamExistence() {
    const [managedStreams, unmanagedStreams] = await Promise.all([this.getManagedStreams(), this.getUnmanagedDataStreams()]);
    const allDefinitionsById = new Map(managedStreams.map(stream => [stream.name, {
      stream,
      exists: false
    }]));
    unmanagedStreams.forEach(stream => {
      if (!allDefinitionsById.get(stream.name)) {
        allDefinitionsById.set(stream.name, {
          stream,
          exists: true
        });
      } else {
        allDefinitionsById.set(stream.name, {
          ...allDefinitionsById.get(stream.name),
          exists: true
        });
      }
    });
    return Array.from(allDefinitionsById.values());
  }

  /**
   * Lists all unmanaged streams (unwired streams without a
   * stored definition).
   */
  async getUnmanagedDataStreams() {
    const response = await wrapEsCall(this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream());
    return response.data_streams.map(dataStream => ({
      name: dataStream.name,
      description: '',
      ingest: {
        lifecycle: {
          inherit: {}
        },
        processing: [],
        unwired: {}
      }
    }));
  }

  /**
   * Lists managed streams, and verifies access to it.
   */
  async getManagedStreams({
    query
  } = {}) {
    const {
      scopedClusterClient,
      storageClient
    } = this.dependencies;
    const streamsSearchResponse = await storageClient.search({
      size: 10000,
      sort: [{
        name: 'asc'
      }],
      track_total_hits: false,
      query
    });
    const streams = streamsSearchResponse.hits.hits.flatMap(hit => this.getStreamDefinitionFromSource(hit._source));
    const privileges = await (0, _stream_crud.checkAccessBulk)({
      names: streams.filter(stream => !_streamsSchema.Streams.GroupStream.Definition.is(stream)).map(stream => stream.name),
      scopedClusterClient
    });
    return streams.filter(stream => {
      var _privileges$stream$na;
      if (_streamsSchema.Streams.GroupStream.Definition.is(stream)) return true;
      return ((_privileges$stream$na = privileges[stream.name]) === null || _privileges$stream$na === void 0 ? void 0 : _privileges$stream$na.read) === true;
    });
  }

  /**
   * Deletes a stream, and its Elasticsearch objects, and its data.
   * Also verifies whether the user has access to the stream.
   */
  async deleteStream(name) {
    const definition = await this.getStream(name);
    if (_streamsSchema.Streams.WiredStream.Definition.is(definition) && (0, _streamsSchema.getParentId)(name) === undefined) {
      throw new _status_error.StatusError('Cannot delete root stream', 400);
    }
    const access = definition && _streamsSchema.Streams.GroupStream.Definition.is(definition) ? {
      write: true,
      read: true
    } : await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient: this.dependencies.scopedClusterClient
    });

    // Can/should State manage access control as well?
    if (!access.write) {
      throw new _security_error.SecurityError(`Cannot delete stream, insufficient privileges`);
    }
    const result = await _state.State.attemptChanges([{
      type: 'delete',
      name
    }], {
      ...this.dependencies,
      streamsClient: this
    });
    if (result.status === 'failed_with_rollback') {
      throw result.error;
    }
    await this.dependencies.queryClient.syncQueries(name, []);
    return {
      acknowledged: true,
      result: 'deleted'
    };
  }
  async updateStoredStream(definition) {
    return this.dependencies.storageClient.index({
      id: definition.name,
      document: definition
    });
  }
  async getAncestors(name) {
    const ancestorIds = (0, _streamsSchema.getAncestors)(name);
    return this.getManagedStreams({
      query: {
        bool: {
          filter: [{
            terms: {
              name: ancestorIds
            }
          }]
        }
      }
    }).then(streams => streams.filter(_streamsSchema.Streams.WiredStream.Definition.is));
  }
  async getDescendants(name) {
    return this.getManagedStreams({
      query: {
        bool: {
          filter: [{
            prefix: {
              name
            }
          }],
          must_not: [{
            term: {
              name
            }
          }]
        }
      }
    }).then(streams => streams.filter(_streamsSchema.Streams.WiredStream.Definition.is));
  }
}
exports.StreamsClient = StreamsClient;