"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetPreviewData = void 0;
var _calculateAuto = require("@kbn/calculate-auto");
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
var _services = require("../domain/services");
var _queries = require("../utils/queries");
var _aggregations = require("./aggregations");
var _transform_generators = require("./transform_generators");
var _synthetics_availability = require("./transform_generators/synthetics_availability");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RANGE_DURATION_24HOURS_LIMIT = 24 * 60 * 60 * 1000 + 60 * 1000; // 24 hours and 1min in milliseconds

class GetPreviewData {
  constructor(esClient, spaceId, dataViewService) {
    this.esClient = esClient;
    this.spaceId = spaceId;
    this.dataViewService = dataViewService;
  }
  async buildRuntimeMappings({
    dataViewId
  }) {
    var _dataView$getRuntimeM, _dataView, _dataView$getRuntimeM2;
    let dataView;
    if (dataViewId) {
      try {
        dataView = await this.dataViewService.get(dataViewId);
      } catch (e) {
        // If the data view is not found, we will continue without it
      }
    }
    return (_dataView$getRuntimeM = (_dataView = dataView) === null || _dataView === void 0 ? void 0 : (_dataView$getRuntimeM2 = _dataView.getRuntimeMappings) === null || _dataView$getRuntimeM2 === void 0 ? void 0 : _dataView$getRuntimeM2.call(_dataView)) !== null && _dataView$getRuntimeM !== void 0 ? _dataView$getRuntimeM : {};
  }
  addExtraTermsOrMultiTermsAgg(perInterval, groupBy) {
    if (!groupBy || groupBy.length === 0) return {
      perInterval
    };
    if (groupBy.length === 1) {
      return {
        perGroup: {
          terms: {
            size: 5,
            field: groupBy[0]
          },
          aggs: {
            perInterval
          }
        },
        perInterval
      };
    }
    return {
      perGroup: {
        multi_terms: {
          size: 5,
          terms: groupBy.map(group => ({
            field: group
          }))
        },
        aggs: {
          perInterval
        }
      },
      perInterval
    };
  }
  async getAPMTransactionDurationPreviewData(indicator, options) {
    var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5;
    const filter = [];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    if (indicator.params.service !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.name': indicator.params.service
      }
    });
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.environment': indicator.params.environment
      }
    });
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.name': indicator.params.transactionName
      }
    });
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.type': indicator.params.transactionType
      }
    });
    if (!!indicator.params.filter) filter.push((0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter));
    const truncatedThreshold = Math.trunc(indicator.params.threshold * 1000);
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await (0, _queries.typedSearch)(this.esClient, {
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            terms: {
              'processor.event': ['metric']
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            exists: {
              field: 'transaction.duration.histogram'
            }
          }, ...filter]
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: '@timestamp',
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          _good: {
            range: {
              field: 'transaction.duration.histogram',
              keyed: true,
              ranges: [{
                to: truncatedThreshold,
                key: 'target'
              }]
            }
          },
          good: {
            bucket_script: {
              buckets_path: {
                _good: `_good['target']>_count`
              },
              script: 'params._good'
            }
          },
          total: {
            value_count: {
              field: 'transaction.duration.histogram'
            }
          }
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore
    (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.perInterval.buckets.map(bucket => {
      var _bucket$good$value, _bucket$good, _bucket$total$value, _bucket$total;
      const good = (_bucket$good$value = (_bucket$good = bucket.good) === null || _bucket$good === void 0 ? void 0 : _bucket$good.value) !== null && _bucket$good$value !== void 0 ? _bucket$good$value : 0;
      const total = (_bucket$total$value = (_bucket$total = bucket.total) === null || _bucket$total === void 0 ? void 0 : _bucket$total.value) !== null && _bucket$total$value !== void 0 ? _bucket$total$value : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          total,
          bad: total - good
        }
      };
    })) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];

    // @ts-ignore
    const groups = (_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : (_response$aggregation4 = _response$aggregation3.perGroup) === null || _response$aggregation4 === void 0 ? void 0 : (_response$aggregation5 = _response$aggregation4.buckets) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$value2, _bucket$good2, _bucket$total$value2, _bucket$total2;
        const good = (_bucket$good$value2 = (_bucket$good2 = bucket.good) === null || _bucket$good2 === void 0 ? void 0 : _bucket$good2.value) !== null && _bucket$good$value2 !== void 0 ? _bucket$good$value2 : 0;
        const total = (_bucket$total$value2 = (_bucket$total2 = bucket.total) === null || _bucket$total2 === void 0 ? void 0 : _bucket$total2.value) !== null && _bucket$total$value2 !== void 0 ? _bucket$total$value2 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async getAPMTransactionErrorPreviewData(indicator, options) {
    var _response$aggregation6, _response$aggregation7, _response$aggregation8, _response$aggregation9, _response$aggregation10;
    const filter = [];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    if (indicator.params.service !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.name': indicator.params.service
      }
    });
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'service.environment': indicator.params.environment
      }
    });
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.name': indicator.params.transactionName
      }
    });
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) filter.push({
      match: {
        'transaction.type': indicator.params.transactionType
      }
    });
    if (!!indicator.params.filter) filter.push((0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter));
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await (0, _queries.typedSearch)(this.esClient, {
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            terms: {
              'event.outcome': ['success', 'failure']
            }
          }, ...filter]
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: '@timestamp',
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          good: {
            filter: {
              bool: {
                should: {
                  match: {
                    'event.outcome': 'success'
                  }
                }
              }
            }
          },
          total: {
            filter: {
              match_all: {}
            }
          }
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore
    (_response$aggregation6 = (_response$aggregation7 = response.aggregations) === null || _response$aggregation7 === void 0 ? void 0 : _response$aggregation7.perInterval.buckets.map(bucket => {
      var _bucket$good$doc_coun, _bucket$good3, _bucket$total$doc_cou, _bucket$total3;
      const good = (_bucket$good$doc_coun = (_bucket$good3 = bucket.good) === null || _bucket$good3 === void 0 ? void 0 : _bucket$good3.doc_count) !== null && _bucket$good$doc_coun !== void 0 ? _bucket$good$doc_coun : 0;
      const total = (_bucket$total$doc_cou = (_bucket$total3 = bucket.total) === null || _bucket$total3 === void 0 ? void 0 : _bucket$total3.doc_count) !== null && _bucket$total$doc_cou !== void 0 ? _bucket$total$doc_cou : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad: total - good,
          total
        }
      };
    })) !== null && _response$aggregation6 !== void 0 ? _response$aggregation6 : [];

    // @ts-ignore
    const groups = (_response$aggregation8 = response.aggregations) === null || _response$aggregation8 === void 0 ? void 0 : (_response$aggregation9 = _response$aggregation8.perGroup) === null || _response$aggregation9 === void 0 ? void 0 : (_response$aggregation10 = _response$aggregation9.buckets) === null || _response$aggregation10 === void 0 ? void 0 : _response$aggregation10.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$doc_coun2, _bucket$good4, _bucket$total$doc_cou2, _bucket$total4;
        const good = (_bucket$good$doc_coun2 = (_bucket$good4 = bucket.good) === null || _bucket$good4 === void 0 ? void 0 : _bucket$good4.doc_count) !== null && _bucket$good$doc_coun2 !== void 0 ? _bucket$good$doc_coun2 : 0;
        const total = (_bucket$total$doc_cou2 = (_bucket$total4 = bucket.total) === null || _bucket$total4 === void 0 ? void 0 : _bucket$total4.doc_count) !== null && _bucket$total$doc_cou2 !== void 0 ? _bucket$total$doc_cou2 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async getHistogramPreviewData(indicator, options) {
    var _response$aggregation11, _response$aggregation12, _response$aggregation13, _response$aggregation14, _response$aggregation15;
    const getHistogramIndicatorAggregations = new _aggregations.GetHistogramIndicatorAggregation(indicator);
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const timestampField = indicator.params.timestampField;
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await this.esClient.search({
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: timestampField,
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          ...getHistogramIndicatorAggregations.execute({
            type: 'good',
            aggregationKey: 'good'
          }),
          ...getHistogramIndicatorAggregations.execute({
            type: 'total',
            aggregationKey: 'total'
          })
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore buckets not typed properly
    (_response$aggregation11 = (_response$aggregation12 = response.aggregations) === null || _response$aggregation12 === void 0 ? void 0 : _response$aggregation12.perInterval.buckets.map(bucket => {
      var _bucket$good$value3, _bucket$good5, _bucket$total$value3, _bucket$total5;
      const good = (_bucket$good$value3 = (_bucket$good5 = bucket.good) === null || _bucket$good5 === void 0 ? void 0 : _bucket$good5.value) !== null && _bucket$good$value3 !== void 0 ? _bucket$good$value3 : 0;
      const total = (_bucket$total$value3 = (_bucket$total5 = bucket.total) === null || _bucket$total5 === void 0 ? void 0 : _bucket$total5.value) !== null && _bucket$total$value3 !== void 0 ? _bucket$total$value3 : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad: total - good,
          total
        }
      };
    })) !== null && _response$aggregation11 !== void 0 ? _response$aggregation11 : [];

    // @ts-ignore
    const groups = (_response$aggregation13 = response.aggregations) === null || _response$aggregation13 === void 0 ? void 0 : (_response$aggregation14 = _response$aggregation13.perGroup) === null || _response$aggregation14 === void 0 ? void 0 : (_response$aggregation15 = _response$aggregation14.buckets) === null || _response$aggregation15 === void 0 ? void 0 : _response$aggregation15.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$value4, _bucket$good6, _bucket$total$value4, _bucket$total6;
        const good = (_bucket$good$value4 = (_bucket$good6 = bucket.good) === null || _bucket$good6 === void 0 ? void 0 : _bucket$good6.value) !== null && _bucket$good$value4 !== void 0 ? _bucket$good$value4 : 0;
        const total = (_bucket$total$value4 = (_bucket$total6 = bucket.total) === null || _bucket$total6 === void 0 ? void 0 : _bucket$total6.value) !== null && _bucket$total$value4 !== void 0 ? _bucket$total$value4 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async getCustomMetricPreviewData(indicator, options) {
    var _response$aggregation16, _response$aggregation17, _response$aggregation18, _response$aggregation19, _response$aggregation20;
    const timestampField = indicator.params.timestampField;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const getCustomMetricIndicatorAggregation = new _aggregations.GetCustomMetricIndicatorAggregation(indicator);
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await this.esClient.search({
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: timestampField,
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          ...getCustomMetricIndicatorAggregation.execute({
            type: 'good',
            aggregationKey: 'good'
          }),
          ...getCustomMetricIndicatorAggregation.execute({
            type: 'total',
            aggregationKey: 'total'
          })
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore buckets not typed properly
    (_response$aggregation16 = (_response$aggregation17 = response.aggregations) === null || _response$aggregation17 === void 0 ? void 0 : _response$aggregation17.perInterval.buckets.map(bucket => {
      var _bucket$good$value5, _bucket$good7, _bucket$total$value5, _bucket$total7;
      const good = (_bucket$good$value5 = (_bucket$good7 = bucket.good) === null || _bucket$good7 === void 0 ? void 0 : _bucket$good7.value) !== null && _bucket$good$value5 !== void 0 ? _bucket$good$value5 : 0;
      const total = (_bucket$total$value5 = (_bucket$total7 = bucket.total) === null || _bucket$total7 === void 0 ? void 0 : _bucket$total7.value) !== null && _bucket$total$value5 !== void 0 ? _bucket$total$value5 : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad: total - good,
          total
        }
      };
    })) !== null && _response$aggregation16 !== void 0 ? _response$aggregation16 : [];

    // @ts-ignore
    const groups = (_response$aggregation18 = response.aggregations) === null || _response$aggregation18 === void 0 ? void 0 : (_response$aggregation19 = _response$aggregation18.perGroup) === null || _response$aggregation19 === void 0 ? void 0 : (_response$aggregation20 = _response$aggregation19.buckets) === null || _response$aggregation20 === void 0 ? void 0 : _response$aggregation20.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$value6, _bucket$good8, _bucket$total$value6, _bucket$total8;
        const good = (_bucket$good$value6 = (_bucket$good8 = bucket.good) === null || _bucket$good8 === void 0 ? void 0 : _bucket$good8.value) !== null && _bucket$good$value6 !== void 0 ? _bucket$good$value6 : 0;
        const total = (_bucket$total$value6 = (_bucket$total8 = bucket.total) === null || _bucket$total8 === void 0 ? void 0 : _bucket$total8.value) !== null && _bucket$total$value6 !== void 0 ? _bucket$total$value6 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async getTimesliceMetricPreviewData(indicator, options) {
    var _response$aggregation21, _response$aggregation22, _response$aggregation23, _response$aggregation24, _response$aggregation25;
    const timestampField = indicator.params.timestampField;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const getCustomMetricIndicatorAggregation = new _aggregations.GetTimesliceMetricIndicatorAggregation(indicator);
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await this.esClient.search({
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: timestampField,
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: getCustomMetricIndicatorAggregation.execute('metric')
      }, options.groupBy)
    });
    const results = // @ts-ignore buckets not typed properly
    (_response$aggregation21 = (_response$aggregation22 = response.aggregations) === null || _response$aggregation22 === void 0 ? void 0 : _response$aggregation22.perInterval.buckets.map(bucket => {
      var _bucket$metric$value, _bucket$metric;
      return {
        date: bucket.key_as_string,
        sliValue: (_bucket$metric$value = (_bucket$metric = bucket.metric) === null || _bucket$metric === void 0 ? void 0 : _bucket$metric.value) !== null && _bucket$metric$value !== void 0 ? _bucket$metric$value : null
      };
    })) !== null && _response$aggregation21 !== void 0 ? _response$aggregation21 : [];

    // @ts-ignore
    const groups = (_response$aggregation23 = response.aggregations) === null || _response$aggregation23 === void 0 ? void 0 : (_response$aggregation24 = _response$aggregation23.perGroup) === null || _response$aggregation24 === void 0 ? void 0 : (_response$aggregation25 = _response$aggregation24.buckets) === null || _response$aggregation25 === void 0 ? void 0 : _response$aggregation25.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$metric$value2, _bucket$metric2;
        return {
          date: bucket.key_as_string,
          sliValue: (_bucket$metric$value2 = (_bucket$metric2 = bucket.metric) === null || _bucket$metric2 === void 0 ? void 0 : _bucket$metric2.value) !== null && _bucket$metric$value2 !== void 0 ? _bucket$metric$value2 : null
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async getCustomKQLPreviewData(indicator, options) {
    var _response$aggregation26, _response$aggregation27, _response$aggregation28, _response$aggregation29, _response$aggregation30;
    const filterQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.filter);
    const goodQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.good);
    const totalQuery = (0, _transform_generators.getElasticsearchQueryOrThrow)(indicator.params.total);
    const timestampField = indicator.params.timestampField;
    const filter = [{
      range: {
        [timestampField]: {
          gte: options.range.start,
          lte: options.range.end
        }
      }
    }, filterQuery];
    const groupingFilters = this.getGroupingFilters(options);
    if (groupingFilters) {
      filter.push(...groupingFilters);
    }
    const index = options.remoteName ? `${options.remoteName}:${indicator.params.index}` : indicator.params.index;
    const response = await (0, _queries.typedSearch)(this.esClient, {
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: timestampField,
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          good: {
            filter: goodQuery
          },
          total: {
            filter: totalQuery
          }
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore
    (_response$aggregation26 = (_response$aggregation27 = response.aggregations) === null || _response$aggregation27 === void 0 ? void 0 : _response$aggregation27.perInterval.buckets.map(bucket => {
      var _bucket$good$doc_coun3, _bucket$good9, _bucket$total$doc_cou3, _bucket$total9;
      const good = (_bucket$good$doc_coun3 = (_bucket$good9 = bucket.good) === null || _bucket$good9 === void 0 ? void 0 : _bucket$good9.doc_count) !== null && _bucket$good$doc_coun3 !== void 0 ? _bucket$good$doc_coun3 : 0;
      const total = (_bucket$total$doc_cou3 = (_bucket$total9 = bucket.total) === null || _bucket$total9 === void 0 ? void 0 : _bucket$total9.doc_count) !== null && _bucket$total$doc_cou3 !== void 0 ? _bucket$total$doc_cou3 : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad: total - good,
          total
        }
      };
    })) !== null && _response$aggregation26 !== void 0 ? _response$aggregation26 : [];

    // @ts-ignore
    const groups = (_response$aggregation28 = response.aggregations) === null || _response$aggregation28 === void 0 ? void 0 : (_response$aggregation29 = _response$aggregation28.perGroup) === null || _response$aggregation29 === void 0 ? void 0 : (_response$aggregation30 = _response$aggregation29.buckets) === null || _response$aggregation30 === void 0 ? void 0 : _response$aggregation30.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$doc_coun4, _bucket$good10, _bucket$total$doc_cou4, _bucket$total10;
        const good = (_bucket$good$doc_coun4 = (_bucket$good10 = bucket.good) === null || _bucket$good10 === void 0 ? void 0 : _bucket$good10.doc_count) !== null && _bucket$good$doc_coun4 !== void 0 ? _bucket$good$doc_coun4 : 0;
        const total = (_bucket$total$doc_cou4 = (_bucket$total10 = bucket.total) === null || _bucket$total10 === void 0 ? void 0 : _bucket$total10.doc_count) !== null && _bucket$total$doc_cou4 !== void 0 ? _bucket$total$doc_cou4 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  getGroupingFilters(options) {
    var _options$groupings;
    const groupingsKeys = Object.keys((_options$groupings = options.groupings) !== null && _options$groupings !== void 0 ? _options$groupings : {});
    if (groupingsKeys.length) {
      return groupingsKeys.map(key => ({
        term: {
          [key]: options.groupings[key]
        }
      }));
    }
  }
  async getSyntheticsAvailabilityPreviewData(indicator, options) {
    var _response$aggregation31, _response$aggregation32, _response$aggregation33, _response$aggregation34, _response$aggregation35;
    const filter = [];
    const {
      monitorIds,
      tags,
      projects
    } = (0, _synthetics_availability.buildParamValues)({
      monitorIds: indicator.params.monitorIds || [],
      tags: indicator.params.tags || [],
      projects: indicator.params.projects || []
    });
    if (!monitorIds.includes(_sloSchema.ALL_VALUE) && monitorIds.length > 0) filter.push({
      terms: {
        'monitor.id': monitorIds
      }
    });
    if (!tags.includes(_sloSchema.ALL_VALUE) && tags.length > 0) filter.push({
      terms: {
        tags
      }
    });
    if (!projects.includes(_sloSchema.ALL_VALUE) && projects.length > 0) filter.push({
      terms: {
        'monitor.project.id': projects
      }
    });
    const index = options.remoteName ? `${options.remoteName}:${_constants.SYNTHETICS_INDEX_PATTERN}` : _constants.SYNTHETICS_INDEX_PATTERN;
    const response = await (0, _queries.typedSearch)(this.esClient, {
      index,
      runtime_mappings: await this.buildRuntimeMappings({
        dataViewId: indicator.params.dataViewId
      }),
      size: 0,
      query: {
        bool: {
          filter: [{
            range: {
              '@timestamp': {
                gte: options.range.start,
                lte: options.range.end
              }
            }
          }, {
            term: {
              'summary.final_attempt': true
            }
          }, {
            term: {
              'meta.space_id': this.spaceId
            }
          }, ...filter]
        }
      },
      aggs: this.addExtraTermsOrMultiTermsAgg({
        date_histogram: {
          field: '@timestamp',
          fixed_interval: options.interval,
          extended_bounds: {
            min: options.range.start,
            max: options.range.end
          }
        },
        aggs: {
          good: {
            filter: {
              term: {
                'monitor.status': 'up'
              }
            }
          },
          total: {
            filter: {
              match_all: {}
            }
          }
        }
      }, options.groupBy)
    });
    const results = // @ts-ignore
    (_response$aggregation31 = (_response$aggregation32 = response.aggregations) === null || _response$aggregation32 === void 0 ? void 0 : _response$aggregation32.perInterval.buckets.map(bucket => {
      var _bucket$good$doc_coun5, _bucket$good11, _bucket$total$doc_cou5, _bucket$total11;
      const good = (_bucket$good$doc_coun5 = (_bucket$good11 = bucket.good) === null || _bucket$good11 === void 0 ? void 0 : _bucket$good11.doc_count) !== null && _bucket$good$doc_coun5 !== void 0 ? _bucket$good$doc_coun5 : 0;
      const total = (_bucket$total$doc_cou5 = (_bucket$total11 = bucket.total) === null || _bucket$total11 === void 0 ? void 0 : _bucket$total11.doc_count) !== null && _bucket$total$doc_cou5 !== void 0 ? _bucket$total$doc_cou5 : 0;
      return {
        date: bucket.key_as_string,
        sliValue: (0, _services.computeSLIForPreview)(good, total),
        events: {
          good,
          bad: total - good,
          total
        }
      };
    })) !== null && _response$aggregation31 !== void 0 ? _response$aggregation31 : [];

    // @ts-ignore
    const groups = (_response$aggregation33 = response.aggregations) === null || _response$aggregation33 === void 0 ? void 0 : (_response$aggregation34 = _response$aggregation33.perGroup) === null || _response$aggregation34 === void 0 ? void 0 : (_response$aggregation35 = _response$aggregation34.buckets) === null || _response$aggregation35 === void 0 ? void 0 : _response$aggregation35.reduce((acc, group) => {
      // @ts-ignore
      acc[group.key] = group.perInterval.buckets.map(bucket => {
        var _bucket$good$doc_coun6, _bucket$good12, _bucket$total$doc_cou6, _bucket$total12;
        const good = (_bucket$good$doc_coun6 = (_bucket$good12 = bucket.good) === null || _bucket$good12 === void 0 ? void 0 : _bucket$good12.doc_count) !== null && _bucket$good$doc_coun6 !== void 0 ? _bucket$good$doc_coun6 : 0;
        const total = (_bucket$total$doc_cou6 = (_bucket$total12 = bucket.total) === null || _bucket$total12 === void 0 ? void 0 : _bucket$total12.doc_count) !== null && _bucket$total$doc_cou6 !== void 0 ? _bucket$total$doc_cou6 : 0;
        return {
          date: bucket.key_as_string,
          sliValue: (0, _services.computeSLIForPreview)(good, total),
          events: {
            good,
            bad: total - good,
            total
          }
        };
      });
      return acc;
    }, {});
    return {
      results,
      groups
    };
  }
  async execute(params) {
    try {
      var _params$objective, _calculateAuto$near$a, _calculateAuto$near, _params$groupBy;
      // If the time range is 24h or less, then we want to use the timeslice duration for the buckets
      // so that the chart is as close to the evaluation as possible.
      // Otherwise due to how the statistics work, the values might not look like
      // they've breached the threshold.
      const rangeDuration = (0, _moment.default)(params.range.to).diff(params.range.from, 'ms');
      const bucketSize = rangeDuration <= RANGE_DURATION_24HOURS_LIMIT && (_params$objective = params.objective) !== null && _params$objective !== void 0 && _params$objective.timesliceWindow ? params.objective.timesliceWindow.asMinutes() : Math.max((_calculateAuto$near$a = (_calculateAuto$near = _calculateAuto.calculateAuto.near(100, _moment.default.duration(rangeDuration, 'ms'))) === null || _calculateAuto$near === void 0 ? void 0 : _calculateAuto$near.asMinutes()) !== null && _calculateAuto$near$a !== void 0 ? _calculateAuto$near$a : 0, 1);
      const options = {
        range: {
          start: params.range.from.getTime(),
          end: params.range.to.getTime()
        },
        remoteName: params.remoteName,
        groupings: params.groupings,
        interval: `${bucketSize}m`,
        groupBy: (_params$groupBy = params.groupBy) === null || _params$groupBy === void 0 ? void 0 : _params$groupBy.filter(value => value !== _sloSchema.ALL_VALUE)
      };
      const type = params.indicator.type;
      switch (type) {
        case 'sli.apm.transactionDuration':
          return this.getAPMTransactionDurationPreviewData(params.indicator, options);
        case 'sli.apm.transactionErrorRate':
          return this.getAPMTransactionErrorPreviewData(params.indicator, options);
        case 'sli.synthetics.availability':
          return this.getSyntheticsAvailabilityPreviewData(params.indicator, options);
        case 'sli.kql.custom':
          return this.getCustomKQLPreviewData(params.indicator, options);
        case 'sli.histogram.custom':
          return this.getHistogramPreviewData(params.indicator, options);
        case 'sli.metric.custom':
          return this.getCustomMetricPreviewData(params.indicator, options);
        case 'sli.metric.timeslice':
          return this.getTimesliceMetricPreviewData(params.indicator, options);
        default:
          (0, _std.assertNever)(type);
      }
    } catch (err) {
      return {
        results: []
      };
    }
  }
}
exports.GetPreviewData = GetPreviewData;