"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateSLO = validateSLO;
var _sloSchema = require("@kbn/slo-schema");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Asserts the SLO Definition is valid from a business invariants point of view.
 * e.g. a 'target' objective requires a number between ]0, 1]
 * e.g. a 'timeslices' budgeting method requires an objective's timeslice_target to be defined.
 *
 * @param slo {SLODefinition}
 */
function validateSLO(slo) {
  if (!isValidId(slo.id)) {
    throw new _errors.IllegalArgumentError('Invalid id');
  }
  if (!isValidObjectiveTarget(slo.objective.target)) {
    throw new _errors.IllegalArgumentError('Invalid objective.target');
  }
  if (_sloSchema.rollingTimeWindowSchema.is(slo.timeWindow) && !isValidRollingTimeWindowDuration(slo.timeWindow.duration)) {
    throw new _errors.IllegalArgumentError('Invalid time_window.duration');
  }
  if (_sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow) && !isValidCalendarAlignedTimeWindowDuration(slo.timeWindow.duration)) {
    throw new _errors.IllegalArgumentError('Invalid time_window.duration');
  }
  if (_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
    if (slo.objective.timesliceTarget === undefined || !isValidTimesliceTarget(slo.objective.timesliceTarget)) {
      throw new _errors.IllegalArgumentError('Invalid objective.timeslice_target');
    }
    if (slo.objective.timesliceWindow === undefined || !isValidTimesliceWindowDuration(slo.objective.timesliceWindow, slo.timeWindow.duration)) {
      throw new _errors.IllegalArgumentError('Invalid objective.timeslice_window');
    }
  }
  validateSettings(slo);
}
function validateSettings(slo) {
  if (!isValidFrequencySettings(slo.settings.frequency)) {
    throw new _errors.IllegalArgumentError('Invalid settings.frequency');
  }
  if (!isValidSyncDelaySettings(slo.settings.syncDelay)) {
    throw new _errors.IllegalArgumentError('Invalid settings.sync_delay');
  }
}
function isValidId(id) {
  const MIN_ID_LENGTH = 8;
  const MAX_ID_LENGTH = 48;
  return MIN_ID_LENGTH <= id.length && id.length <= MAX_ID_LENGTH;
}
function isValidObjectiveTarget(value) {
  return value > 0 && value < 1;
}
function isValidTimesliceTarget(value) {
  return value >= 0 && value <= 1;
}
function isValidRollingTimeWindowDuration(duration) {
  // 7, 30 or 90days accepted
  return duration.unit === _sloSchema.DurationUnit.Day && [7, 30, 90].includes(duration.value);
}
function isValidCalendarAlignedTimeWindowDuration(duration) {
  // 1 week or 1 month
  return [_sloSchema.DurationUnit.Week, _sloSchema.DurationUnit.Month].includes(duration.unit) && duration.value === 1;
}
function isValidTimesliceWindowDuration(timesliceWindow, timeWindow) {
  return [_sloSchema.DurationUnit.Minute, _sloSchema.DurationUnit.Hour].includes(timesliceWindow.unit) && timesliceWindow.isShorterThan(timeWindow);
}

/**
 * validate that 1 minute <= frequency < 1 hour
 */
function isValidFrequencySettings(frequency) {
  return frequency.isLongerOrEqualThan(new _sloSchema.Duration(1, _sloSchema.DurationUnit.Minute)) && frequency.isShorterThan(new _sloSchema.Duration(1, _sloSchema.DurationUnit.Hour));
}

/**
 * validate that 1 minute <= sync_delay < 6 hour
 */
function isValidSyncDelaySettings(syncDelay) {
  return syncDelay.isLongerOrEqualThan(new _sloSchema.Duration(1, _sloSchema.DurationUnit.Minute)) && syncDelay.isShorterThan(new _sloSchema.Duration(6, _sloSchema.DurationUnit.Hour));
}