"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateBurnRateRule = validateBurnRateRule;
var _i18n = require("@kbn/i18n");
var _duration = require("../../utils/slo/duration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MIN_DURATION_IN_HOURS = 1;
const MAX_DURATION_IN_HOURS = 72;
function validateBurnRateRule(ruleParams) {
  const validationResult = {
    errors: {
      sloId: new Array(),
      windows: new Array()
    }
  };
  const {
    sloId,
    windows
  } = ruleParams;
  if (!sloId) {
    validationResult.errors.sloId.push(SLO_REQUIRED);
  }
  if (windows) {
    windows.forEach(({
      burnRateThreshold,
      longWindow,
      shortWindow,
      maxBurnRateThreshold
    }) => {
      const result = {
        longWindow: new Array(),
        shortWindow: new Array(),
        burnRateThreshold: new Array()
      };
      if (burnRateThreshold === undefined || maxBurnRateThreshold === undefined) {
        result.burnRateThreshold.push(BURN_RATE_THRESHOLD_REQUIRED);
      } else if (sloId && (burnRateThreshold < 0.01 || burnRateThreshold > maxBurnRateThreshold)) {
        result.burnRateThreshold.push(getInvalidThresholdValueError(maxBurnRateThreshold));
      }
      if (longWindow === undefined) {
        result.longWindow.push(LONG_WINDOW_DURATION_REQUIRED);
      } else if (!isValidLongWindowDuration(longWindow)) {
        result.longWindow.push(LONG_WINDOW_DURATION_INVALID);
      }
      if (shortWindow === undefined) {
        result.shortWindow.push(SHORT_WINDOW_DURATION_REQUIRED);
      } else if (!isValidShortWindowDuration(shortWindow, longWindow)) {
        result.shortWindow.push(SHORT_WINDOW_DURATION_INVALID);
      }
      validationResult.errors.windows.push(result);
    });
  }
  return validationResult;
}
const isValidLongWindowDuration = duration => {
  const {
    unit,
    value
  } = duration;
  return unit === 'h' && value >= MIN_DURATION_IN_HOURS && value <= MAX_DURATION_IN_HOURS;
};
const SLO_REQUIRED = _i18n.i18n.translate('xpack.slo.rules.burnRate.errors.sloRequired', {
  defaultMessage: 'SLO is required.'
});
const LONG_WINDOW_DURATION_REQUIRED = _i18n.i18n.translate('xpack.slo.rules.burnRate.errors.windowDurationRequired', {
  defaultMessage: 'The long lookback period is required.'
});
const LONG_WINDOW_DURATION_INVALID = _i18n.i18n.translate('xpack.slo.rules.longWindow.errorText', {
  defaultMessage: 'The lookback period must be between 1 and 72 hours.'
});
const isValidShortWindowDuration = (shortWindow, longWindow) => {
  const longWindowInMinutes = (0, _duration.toMinutes)(longWindow);
  const shortWindowInMinutes = (0, _duration.toMinutes)(shortWindow);
  const {
    unit
  } = shortWindow;
  return shortWindowInMinutes >= 1 && shortWindowInMinutes <= longWindowInMinutes && unit === 'm';
};
const SHORT_WINDOW_DURATION_REQUIRED = _i18n.i18n.translate('xpack.slo.rules.burnRate.errors.shortWindowDurationRequired', {
  defaultMessage: 'The short lookback period is required.'
});
const SHORT_WINDOW_DURATION_INVALID = _i18n.i18n.translate('xpack.slo.rules.shortWindow.errorText', {
  defaultMessage: 'The short lookback period must be lower than the long lookback period.'
});
const BURN_RATE_THRESHOLD_REQUIRED = _i18n.i18n.translate('xpack.slo.rules.burnRate.errors.burnRateThresholdRequired', {
  defaultMessage: 'Burn rate threshold is required.'
});
const getInvalidThresholdValueError = maxBurnRate => _i18n.i18n.translate('xpack.slo.rules.burnRate.errors.invalidThresholdValue', {
  defaultMessage: 'Burn rate threshold must be between 0.01 and {maxBurnRate}.',
  values: {
    maxBurnRate
  }
});