"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionsClientChat = void 0;
var _server = require("@kbn/langchain/server");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const llmTypeDictionary = {
  [`.gen-ai`]: `openai`,
  [`.bedrock`]: `bedrock`,
  [`.gemini`]: `gemini`,
  [`.inference`]: `inference`
};
class ActionsClientChat {
  constructor(actionsClient, logger) {
    this.actionsClient = actionsClient;
    this.logger = logger;
  }
  async createModel({
    migrationId,
    connectorId,
    abortController
  }) {
    var _connector$config;
    const connector = await this.actionsClient.get({
      id: connectorId
    });
    if (!connector) {
      throw new Error(`Connector not found: ${connectorId}`);
    }
    const llmType = this.getLLMType(connector.actionTypeId);
    const ChatModelClass = this.getLLMClass(llmType);
    const model = new ChatModelClass({
      actionsClient: this.actionsClient,
      connectorId,
      llmType,
      model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel,
      streaming: false,
      convertSystemMessageToHumanContent: false,
      temperature: 0.05,
      maxRetries: 1,
      // Only retry once inside the model, we will handle backoff retries in the task runner
      telemetryMetadata: {
        pluginId: _constants.TELEMETRY_SIEM_MIGRATION_ID,
        aggregateBy: migrationId
      },
      signal: abortController.signal,
      logger: this.logger
    });
    return model;
  }
  getLLMType(actionTypeId) {
    if (llmTypeDictionary[actionTypeId]) {
      return llmTypeDictionary[actionTypeId];
    }
    throw new Error(`Unknown LLM type for action type ID: ${actionTypeId}`);
  }
  getLLMClass(llmType) {
    switch (llmType) {
      case 'bedrock':
        return _server.ActionsClientBedrockChatModel;
      case 'gemini':
        return _server.ActionsClientChatVertexAI;
      case 'openai':
      case 'inference':
      default:
        return _server.ActionsClientChatOpenAI;
    }
  }
}
exports.ActionsClientChat = ActionsClientChat;