"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationTaskEvaluator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _evaluation = require("langsmith/evaluation");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _langsmith2 = require("langsmith");
var _fastestLevenshtein = require("fastest-levenshtein");
var _rule_migrations_task_runner = require("./rule_migrations_task_runner");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationTaskEvaluator extends _rule_migrations_task_runner.RuleMigrationTaskRunner {
  constructor(...args) {
    super(...args);
    /**
     * This is a map of custom evaluators that are used to evaluate rule migration tasks
     * The object keys are used for the `key` property of the evaluation result, and the value is a function that takes a the `run` and `example`
     * and returns a `score` and a `comment` (and any other data needed for the evaluation)
     **/
    (0, _defineProperty2.default)(this, "evaluators", {
      translation_result: ({
        run,
        example
      }) => {
        var _run$outputs, _example$outputs;
        const runResult = run === null || run === void 0 ? void 0 : (_run$outputs = run.outputs) === null || _run$outputs === void 0 ? void 0 : _run$outputs.translation_result;
        const expectedResult = example === null || example === void 0 ? void 0 : (_example$outputs = example.outputs) === null || _example$outputs === void 0 ? void 0 : _example$outputs.translation_result;
        if (!expectedResult) {
          return {
            comment: 'No translation result expected'
          };
        }
        if (!runResult) {
          return {
            score: false,
            comment: 'No translation result received'
          };
        }
        if (runResult === expectedResult) {
          return {
            score: true,
            comment: 'Correct'
          };
        }
        return {
          score: false,
          comment: `Incorrect, expected "${expectedResult}" but got "${runResult}"`
        };
      },
      custom_query_accuracy: ({
        run,
        example
      }) => {
        var _run$outputs2, _run$outputs2$elastic, _example$outputs2, _example$outputs2$ela;
        const runQuery = run === null || run === void 0 ? void 0 : (_run$outputs2 = run.outputs) === null || _run$outputs2 === void 0 ? void 0 : (_run$outputs2$elastic = _run$outputs2.elastic_rule) === null || _run$outputs2$elastic === void 0 ? void 0 : _run$outputs2$elastic.query;
        const expectedQuery = example === null || example === void 0 ? void 0 : (_example$outputs2 = example.outputs) === null || _example$outputs2 === void 0 ? void 0 : (_example$outputs2$ela = _example$outputs2.elastic_rule) === null || _example$outputs2$ela === void 0 ? void 0 : _example$outputs2$ela.query;
        if (!expectedQuery) {
          if (runQuery) {
            return {
              score: 0,
              comment: 'No custom translation expected, but received'
            };
          }
          return {
            comment: 'No custom translation expected'
          };
        }
        if (!runQuery) {
          return {
            score: 0,
            comment: 'Custom translation expected, but not received'
          };
        }

        // calculate the levenshtein distance between the two queries:
        // The distance is the minimum number of single-character edits required to change one word into the other.
        // So, the distance is a number between 0 and the length of the longest string.
        const queryDistance = (0, _fastestLevenshtein.distance)(runQuery, expectedQuery);
        const maxDistance = Math.max(expectedQuery.length, runQuery.length);
        // The similarity is a number between 0 and 1 (score), where 1 means the two strings are identical.
        const similarity = 1 - queryDistance / maxDistance;
        return {
          score: Math.round(similarity * 1000) / 1000,
          // round to 3 decimal places
          comment: `Distance: ${queryDistance}`
        };
      },
      prebuilt_rule_match: ({
        run,
        example
      }) => {
        var _run$outputs3, _run$outputs3$elastic, _example$outputs3, _example$outputs3$ela;
        const runPrebuiltRuleId = run === null || run === void 0 ? void 0 : (_run$outputs3 = run.outputs) === null || _run$outputs3 === void 0 ? void 0 : (_run$outputs3$elastic = _run$outputs3.elastic_rule) === null || _run$outputs3$elastic === void 0 ? void 0 : _run$outputs3$elastic.prebuilt_rule_id;
        const expectedPrebuiltRuleId = example === null || example === void 0 ? void 0 : (_example$outputs3 = example.outputs) === null || _example$outputs3 === void 0 ? void 0 : (_example$outputs3$ela = _example$outputs3.elastic_rule) === null || _example$outputs3$ela === void 0 ? void 0 : _example$outputs3$ela.prebuilt_rule_id;
        if (!expectedPrebuiltRuleId) {
          if (runPrebuiltRuleId) {
            return {
              score: false,
              comment: 'No prebuilt rule expected, but received'
            };
          }
          return {
            comment: 'No prebuilt rule expected'
          };
        }
        if (!runPrebuiltRuleId) {
          return {
            score: false,
            comment: 'Prebuilt rule expected, but not received'
          };
        }
        if (runPrebuiltRuleId === expectedPrebuiltRuleId) {
          return {
            score: true,
            comment: 'Correct match'
          };
        }
        return {
          score: false,
          comment: `Incorrect match, expected ID is "${expectedPrebuiltRuleId}" but got "${runPrebuiltRuleId}"`
        };
      }
    });
  }
  async evaluate({
    connectorId,
    langsmithOptions,
    invocationConfig
  }) {
    if (!(0, _langsmith.isLangSmithEnabled)()) {
      throw Error('LangSmith is not enabled');
    }
    const client = new _langsmith2.Client({
      apiKey: langsmithOptions.api_key
    });

    // Make sure the dataset exists
    const dataset = [];
    for await (const example of client.listExamples({
      datasetName: langsmithOptions.dataset
    })) {
      dataset.push(example);
    }
    if (dataset.length === 0) {
      throw Error(`LangSmith dataset not found: ${langsmithOptions.dataset}`);
    }

    // Initialize the the task runner first, this may take some time
    await this.initialize();

    // Check if the connector exists and user has privileges to read it
    const connector = await this.dependencies.actionsClient.get({
      id: connectorId
    });
    if (!connector) {
      throw Error(`Connector with id ${connectorId} not found`);
    }

    // for each connector, setup the evaluator
    await this.setup(connectorId);

    // create the migration task after setup
    const migrateRuleTask = this.createMigrateRuleTask(invocationConfig);
    const evaluators = this.getEvaluators();
    (0, _evaluation.evaluate)(migrateRuleTask, {
      data: langsmithOptions.dataset,
      experimentPrefix: connector.name,
      evaluators,
      client,
      maxConcurrency: 3
    }).then(() => {
      this.logger.info('Evaluation finished');
    }).catch(err => {
      this.logger.error(`Evaluation error:\n ${JSON.stringify(err, null, 2)}`);
    });
  }
  getEvaluators() {
    return Object.entries(this.evaluators).map(([key, evaluator]) => {
      return args => {
        const result = evaluator(args);
        return {
          key,
          ...result
        };
      };
    });
  }
}
exports.RuleMigrationTaskEvaluator = RuleMigrationTaskEvaluator;