"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsDataBaseClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_PIT_KEEP_ALIVE = '30s';
class RuleMigrationsDataBaseClient {
  constructor(getIndexName, currentUser, esScopedClient, logger, dependencies) {
    (0, _defineProperty2.default)(this, "esClient", void 0);
    this.getIndexName = getIndexName;
    this.currentUser = currentUser;
    this.esScopedClient = esScopedClient;
    this.logger = logger;
    this.dependencies = dependencies;
    this.esClient = esScopedClient.asInternalUser;
  }
  async getProfileUid() {
    if (this.currentUser.profile_uid) {
      return this.currentUser.profile_uid;
    }
    const username = this.currentUser.username;
    try {
      const users = await this.esScopedClient.asCurrentUser.security.getUser({
        username,
        with_profile_uid: true
      });
      return users[username].profile_uid;
    } catch (error) {
      this.logger.error(`Error getting profile_uid for user ${username}: ${error}`);
      return username;
    }
  }
  processHit(hit, override = {}) {
    const {
      _id,
      _source
    } = hit;
    (0, _assert.default)(_id, 'document should have _id');
    (0, _assert.default)(_source, 'document should have _source');
    return {
      ..._source,
      ...override,
      id: _id
    };
  }
  processHits(hits = [], override = {}) {
    return hits.map(hit => this.processHit(hit, override));
  }
  processResponseHits(response, override) {
    return this.processHits(response.hits.hits, override);
  }
  getTotalHits(response) {
    var _response$hits$total$, _response$hits$total;
    return typeof response.hits.total === 'number' ? response.hits.total : (_response$hits$total$ = (_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) !== null && _response$hits$total$ !== void 0 ? _response$hits$total$ : 0;
  }

  /** Returns functions to iterate over all the search results in batches */
  getSearchBatches(search, keepAlive = DEFAULT_PIT_KEEP_ALIVE) {
    const pitPromise = this.getIndexName().then(index => this.esClient.openPointInTime({
      index,
      keep_alive: keepAlive
    }).then(({
      id
    }) => ({
      id,
      keep_alive: keepAlive
    })));
    let currentBatchSearch;
    /* Returns the next batch of search results */
    const next = async () => {
      const pit = await pitPromise;
      if (!currentBatchSearch) {
        currentBatchSearch = this.esClient.search({
          ...search,
          pit
        });
      } else {
        currentBatchSearch = currentBatchSearch.then(previousResponse => {
          if (previousResponse.hits.hits.length === 0) {
            return previousResponse;
          }
          const lastSort = previousResponse.hits.hits[previousResponse.hits.hits.length - 1].sort;
          return this.esClient.search({
            ...search,
            pit,
            search_after: lastSort
          });
        });
      }
      const response = await currentBatchSearch;
      return this.processResponseHits(response);
    };

    /** Returns all the search results */
    const all = async () => {
      const allResults = [];
      let results = await next();
      while (results.length) {
        allResults.push(...results);
        results = await next();
      }
      return allResults;
    };
    return {
      next,
      all
    };
  }
}
exports.RuleMigrationsDataBaseClient = RuleMigrationsDataBaseClient;