"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upgradeLatestTransformIfNeeded = exports.stopTransform = exports.scheduleTransformNow = exports.scheduleLatestTransformNow = exports.removeLegacyTransforms = exports.reinstallTransform = exports.getLegacyTransforms = exports.getLatestTransformId = exports.deleteTransform = exports.createTransform = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _search_strategy = require("../../../../common/search_strategy");
var _risk_score_modules = require("../../../../common/utils/risk_score_modules");
var _configurations = require("../risk_score/configurations");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getLegacyTransforms = async ({
  namespace,
  esClient
}) => {
  const getTransformStatsRequests = [];
  [_search_strategy.EntityType.host, _search_strategy.EntityType.user].forEach(entity => {
    getTransformStatsRequests.push(esClient.transform.getTransform({
      transform_id: (0, _risk_score_modules.getRiskScorePivotTransformId)(entity, namespace)
    }));
    getTransformStatsRequests.push(esClient.transform.getTransform({
      transform_id: (0, _risk_score_modules.getRiskScoreLatestTransformId)(entity, namespace)
    }));
  });
  const results = await Promise.allSettled(getTransformStatsRequests);
  const transforms = results.reduce((acc, result) => {
    var _result$value, _result$value$transfo;
    if (result.status === 'fulfilled' && ((_result$value = result.value) === null || _result$value === void 0 ? void 0 : (_result$value$transfo = _result$value.transforms) === null || _result$value$transfo === void 0 ? void 0 : _result$value$transfo.length) > 0) {
      acc.push(...result.value.transforms);
    }
    return acc;
  }, []);
  return transforms;
};
exports.getLegacyTransforms = getLegacyTransforms;
const removeLegacyTransforms = async ({
  namespace,
  esClient
}) => {
  const transforms = await getLegacyTransforms({
    namespace,
    esClient
  });
  const stopTransformRequests = transforms.map(t => esClient.transform.deleteTransform({
    transform_id: t.id,
    force: true
  }));
  await Promise.allSettled(stopTransformRequests);
};
exports.removeLegacyTransforms = removeLegacyTransforms;
const createTransform = async ({
  esClient,
  transform,
  logger
}) => {
  try {
    await esClient.transform.getTransform({
      transform_id: transform.transform_id
    });
    logger.info(`Transform ${transform.transform_id} already exists`);
  } catch (existErr) {
    const transformedError = (0, _securitysolutionEsUtils.transformError)(existErr);
    if (transformedError.statusCode === 404) {
      return esClient.transform.putTransform(transform);
    } else {
      logger.error(`Failed to check if transform ${transform.transform_id} exists before creation: ${transformedError.message}`);
      throw existErr;
    }
  }
};
exports.createTransform = createTransform;
const stopTransform = async ({
  esClient,
  logger,
  transformId
}) => (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.transform.stopTransform({
  transform_id: transformId,
  wait_for_completion: true,
  force: true
}, {
  ignore: [409, 404]
}), {
  logger
});
exports.stopTransform = stopTransform;
const deleteTransform = ({
  esClient,
  logger,
  transformId,
  deleteData = false
}) => (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.transform.deleteTransform({
  transform_id: transformId,
  force: true,
  delete_dest_index: deleteData
}, {
  ignore: [404]
}), {
  logger
});
exports.deleteTransform = deleteTransform;
const reinstallTransform = async ({
  esClient,
  logger,
  config
}) => {
  const transformId = config.transform_id;
  await stopTransform({
    esClient,
    logger,
    transformId
  });
  await deleteTransform({
    esClient,
    logger,
    transformId
  });
  await createTransform({
    esClient,
    logger,
    transform: config
  });
};
exports.reinstallTransform = reinstallTransform;
const getLatestTransformId = namespace => `risk_score_latest_transform_${namespace}`;
exports.getLatestTransformId = getLatestTransformId;
const hasTransformStarted = transformStats => {
  return transformStats.state === 'indexing' || transformStats.state === 'started';
};
const scheduleTransformNow = async ({
  esClient,
  transformId
}) => {
  const transformStats = await esClient.transform.getTransformStats({
    transform_id: transformId
  });
  if (transformStats.count <= 0) {
    throw new Error(`Unable to find transform status for [${transformId}] while attempting to schedule`);
  }
  if (!hasTransformStarted(transformStats.transforms[0])) {
    await esClient.transform.startTransform({
      transform_id: transformId
    });
  } else {
    await esClient.transform.scheduleNowTransform({
      transform_id: transformId
    });
  }
};

/**
 * This method updates the transform configuration if it is outdated.
 * If the 'latest' property of the transform changes it will reinstall the transform.
 */
exports.scheduleTransformNow = scheduleTransformNow;
const upgradeLatestTransformIfNeeded = async ({
  esClient,
  namespace,
  logger
}) => {
  const transformId = getLatestTransformId(namespace);
  const latestIndex = (0, _risk_engine.getRiskScoreLatestIndex)(namespace);
  const timeSeriesIndex = (0, _risk_engine.getRiskScoreTimeSeriesIndex)(namespace);
  const response = await esClient.transform.getTransform({
    transform_id: transformId
  });
  const newConfig = (0, _configurations.getTransformOptions)({
    dest: latestIndex,
    source: [timeSeriesIndex]
  });
  if (isTransformOutdated(response.transforms[0], newConfig)) {
    if (doesTransformRequireReinstall(response.transforms[0], newConfig)) {
      logger.info(`Reinstalling transform ${transformId}`);
      await reinstallTransform({
        esClient,
        logger,
        config: {
          ...newConfig,
          transform_id: transformId
        }
      });
    } else {
      logger.info(`Upgrading transform ${transformId}`);
      const {
        latest: _unused,
        ...changes
      } = newConfig;
      await esClient.transform.updateTransform({
        transform_id: transformId,
        ...changes
      });
    }
  }
};
exports.upgradeLatestTransformIfNeeded = upgradeLatestTransformIfNeeded;
const scheduleLatestTransformNow = async ({
  namespace,
  esClient,
  logger
}) => {
  const transformId = getLatestTransformId(namespace);
  try {
    await upgradeLatestTransformIfNeeded({
      esClient,
      namespace,
      logger
    });
  } catch (err) {
    logger.error(`There was an error upgrading the transform ${transformId}. Continuing with transform scheduling. ${err.message}`);
  }
  await scheduleTransformNow({
    esClient,
    transformId
  });
};
exports.scheduleLatestTransformNow = scheduleLatestTransformNow;
const isTransformOutdated = (transform, newConfig) => {
  var _transform$_meta, _newConfig$_meta;
  return ((_transform$_meta = transform._meta) === null || _transform$_meta === void 0 ? void 0 : _transform$_meta.version) !== ((_newConfig$_meta = newConfig._meta) === null || _newConfig$_meta === void 0 ? void 0 : _newConfig$_meta.version);
};
const doesTransformRequireReinstall = (transform, newConfig) => JSON.stringify(transform.latest) !== JSON.stringify(newConfig.latest);