"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withRiskEnginePrivilegeCheck = exports.getUserRiskEnginePrivileges = exports._getMissingPrivilegesMessage = void 0;
var _routes = require("@kbn/lists-plugin/server/routes");
var _i18n = require("@kbn/i18n");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _check_and_format_privileges = require("../utils/check_and_format_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUserRiskEnginePrivileges = async (request, security) => {
  return (0, _check_and_format_privileges.checkAndFormatPrivileges)({
    request,
    security,
    privilegesToCheck: {
      elasticsearch: {
        cluster: _risk_engine.RISK_ENGINE_REQUIRED_ES_CLUSTER_PRIVILEGES,
        index: _risk_engine.RISK_ENGINE_REQUIRED_ES_INDEX_PRIVILEGES
      }
    }
  });
};
exports.getUserRiskEnginePrivileges = getUserRiskEnginePrivileges;
const _getMissingPrivilegesMessage = riskEnginePrivileges => {
  const {
    indexPrivileges,
    clusterPrivileges
  } = (0, _risk_engine.getMissingRiskEnginePrivileges)(riskEnginePrivileges.privileges);
  const indexPrivilegesMessage = indexPrivileges.map(([indexName, privileges]) => _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.missingIndexPrivilege', {
    defaultMessage: 'Missing index privileges for index "{indexName}": {privileges}.',
    values: {
      indexName,
      privileges: privileges.join(', ')
    }
  })).join('\n');
  const clusterPrivilegesMessage = !clusterPrivileges.length ? '' : _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.missingClusterPrivilege', {
    defaultMessage: 'Missing cluster privileges: {privileges}.',
    values: {
      privileges: clusterPrivileges.join(', ')
    }
  });
  const unauthorizedMessage = _i18n.i18n.translate('xpack.securitySolution.entityAnalytics.riskEngine.unauthorized', {
    defaultMessage: 'User is missing risk engine privileges.'
  });
  return `${unauthorizedMessage} ${indexPrivilegesMessage} ${clusterPrivilegesMessage}`;
};

/**
 * This function is used to check if the user has the required privileges to access the risk engine.
 * It is used to wrap a risk engine route handler which requires full access to the risk engine.
 * @param getStartServices - Kibana's start services accessor
 * @param handler - The route handler to wrap
 **/
exports._getMissingPrivilegesMessage = _getMissingPrivilegesMessage;
const withRiskEnginePrivilegeCheck = (getStartServices, handler) => {
  return async (context, request, response) => {
    const [_, {
      security
    }] = await getStartServices();
    const privileges = await getUserRiskEnginePrivileges(request, security);
    if (!privileges.has_all_required) {
      const siemResponse = (0, _routes.buildSiemResponse)(response);
      return siemResponse.error({
        statusCode: 403,
        body: _getMissingPrivilegesMessage(privileges)
      });
    }
    return handler(context, request, response);
  };
};
exports.withRiskEnginePrivilegeCheck = withRiskEnginePrivilegeCheck;