"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RiskEngineDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _types = require("../../../../common/entity_analytics/types");
var _entity_analytics = require("../../../../common/api/entity_analytics");
var _transforms = require("../utils/transforms");
var _saved_object_configuration = require("./utils/saved_object_configuration");
var _bulk_delete_saved_objects = require("../../risk_score/prebuilt_saved_objects/helpers/bulk_delete_saved_objects");
var _tasks = require("../risk_score/tasks");
var _audit = require("./audit");
var _audit2 = require("../audit");
var _risk_scoring_task = require("../risk_score/tasks/risk_scoring_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RiskEngineDataClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "getConfiguration", () => (0, _saved_object_configuration.getConfiguration)({
      savedObjectsClient: this.options.soClient
    }));
    (0, _defineProperty2.default)(this, "updateConfiguration", config => (0, _saved_object_configuration.updateSavedObjectAttribute)({
      savedObjectsClient: this.options.soClient,
      attributes: config
    }));
    this.options = options;
  }
  async init({
    namespace,
    taskManager,
    riskScoreDataClient
  }) {
    const result = {
      legacyRiskEngineDisabled: false,
      riskEngineResourcesInstalled: false,
      riskEngineConfigurationCreated: false,
      riskEngineEnabled: false,
      errors: []
    };
    try {
      var _this$options$auditLo;
      result.legacyRiskEngineDisabled = await this.disableLegacyRiskEngine({
        namespace
      });
      (_this$options$auditLo = this.options.auditLogger) === null || _this$options$auditLo === void 0 ? void 0 : _this$options$auditLo.log({
        message: 'System disabled the legacy risk engine.',
        event: {
          action: _audit.RiskEngineAuditActions.RISK_ENGINE_DISABLE_LEGACY_ENGINE,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.CHANGE,
          outcome: _audit2.AUDIT_OUTCOME.SUCCESS
        }
      });
    } catch (e) {
      result.legacyRiskEngineDisabled = false;
      result.errors.push(e.message);
    }
    try {
      await riskScoreDataClient.init();
      result.riskEngineResourcesInstalled = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }
    try {
      const soResult = await (0, _saved_object_configuration.initSavedObjects)({
        savedObjectsClient: this.options.soClient,
        namespace
      });
      this.options.logger.info(`Risk engine savedObject configuration: ${JSON.stringify(soResult, null, 2)}`);
      result.riskEngineConfigurationCreated = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }

    // should be the last step, after all resources are installed
    try {
      await this.enableRiskEngine({
        taskManager
      });
      result.riskEngineEnabled = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }
    return result;
  }
  async getStatus({
    namespace,
    taskManager
  }) {
    var _this$options$auditLo2;
    const riskEngineStatus = await this.getCurrentStatus();
    const legacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    const taskStatus = riskEngineStatus === 'ENABLED' && taskManager ? await (0, _risk_scoring_task.getRiskScoringTaskStatus)({
      namespace,
      taskManager
    }) : undefined;
    (_this$options$auditLo2 = this.options.auditLogger) === null || _this$options$auditLo2 === void 0 ? void 0 : _this$options$auditLo2.log({
      message: 'User checked if the risk engine is enabled',
      event: {
        action: _audit.RiskEngineAuditActions.RISK_ENGINE_STATUS_GET,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.ACCESS,
        outcome: _audit2.AUDIT_OUTCOME.SUCCESS
      }
    });
    return {
      riskEngineStatus,
      legacyRiskEngineStatus,
      taskStatus
    };
  }
  async enableRiskEngine({
    taskManager
  }) {
    try {
      var _this$options$auditLo3;
      const configurationResult = await (0, _saved_object_configuration.updateSavedObjectAttribute)({
        savedObjectsClient: this.options.soClient,
        attributes: {
          enabled: true
        }
      });
      await (0, _tasks.startRiskScoringTask)({
        logger: this.options.logger,
        namespace: this.options.namespace,
        riskEngineDataClient: this,
        taskManager
      });
      (_this$options$auditLo3 = this.options.auditLogger) === null || _this$options$auditLo3 === void 0 ? void 0 : _this$options$auditLo3.log({
        message: 'User started risk scoring service',
        event: {
          action: _audit.RiskEngineAuditActions.RISK_ENGINE_START,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.CHANGE,
          outcome: _audit2.AUDIT_OUTCOME.SUCCESS
        }
      });
      return configurationResult;
    } catch (e) {
      var _this$options$auditLo4;
      this.options.logger.error(`Error while enabling risk engine: ${e.message}`);
      (_this$options$auditLo4 = this.options.auditLogger) === null || _this$options$auditLo4 === void 0 ? void 0 : _this$options$auditLo4.log({
        message: 'System stopped risk scoring service after error occurred',
        event: {
          action: _audit.RiskEngineAuditActions.RISK_ENGINE_DISABLE,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.CHANGE,
          outcome: _audit2.AUDIT_OUTCOME.FAILURE
        },
        error: e
      });
      await this.disableRiskEngine({
        taskManager
      });
      throw e;
    }
  }
  async disableRiskEngine({
    taskManager
  }) {
    var _this$options$auditLo5;
    await (0, _tasks.removeRiskScoringTask)({
      namespace: this.options.namespace,
      taskManager,
      logger: this.options.logger
    });
    (_this$options$auditLo5 = this.options.auditLogger) === null || _this$options$auditLo5 === void 0 ? void 0 : _this$options$auditLo5.log({
      message: 'User removed risk scoring task',
      event: {
        action: _audit.RiskEngineAuditActions.RISK_ENGINE_REMOVE_TASK,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.CHANGE,
        outcome: _audit2.AUDIT_OUTCOME.SUCCESS
      }
    });
    return (0, _saved_object_configuration.updateSavedObjectAttribute)({
      savedObjectsClient: this.options.soClient,
      attributes: {
        enabled: false
      }
    });
  }
  async scheduleNow({
    taskManager
  }) {
    var _this$options$auditLo6;
    const riskEngineStatus = await this.getCurrentStatus();
    if (riskEngineStatus !== 'ENABLED') {
      throw new Error(`The risk engine must be enable to schedule a run. Current status: ${riskEngineStatus}`);
    }
    (_this$options$auditLo6 = this.options.auditLogger) === null || _this$options$auditLo6 === void 0 ? void 0 : _this$options$auditLo6.log({
      message: 'User scheduled a risk engine run',
      event: {
        action: _audit.RiskEngineAuditActions.RISK_ENGINE_SCHEDULE_NOW,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.ACCESS,
        outcome: _audit2.AUDIT_OUTCOME.SUCCESS
      }
    });
    return (0, _risk_scoring_task.scheduleNow)({
      taskManager,
      namespace: this.options.namespace,
      logger: this.options.logger
    });
  }

  /**
   * Delete all risk engine resources.
   *
   * It returns an array of errors that occurred during the deletion.
   *
   * WARNING: It will remove all data.
   */
  async tearDown({
    taskManager,
    riskScoreDataClient
  }) {
    const errors = [];
    const addError = e => errors.push(e);
    await (0, _tasks.removeRiskScoringTask)({
      namespace: this.options.namespace,
      taskManager,
      logger: this.options.logger
    }).catch(addError);
    await (0, _saved_object_configuration.deleteSavedObjects)({
      savedObjectsClient: this.options.soClient
    }).catch(addError);
    const riskScoreErrors = await riskScoreDataClient.tearDown();
    return errors.concat(riskScoreErrors);
  }
  async disableLegacyRiskEngine({
    namespace
  }) {
    const legacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    if (legacyRiskEngineStatus === _entity_analytics.RiskEngineStatusEnum.NOT_INSTALLED) {
      return true;
    }
    await (0, _transforms.removeLegacyTransforms)({
      esClient: this.options.esClient,
      namespace
    });
    const deleteDashboardsPromises = [_types.LegacyEntityType.host, _types.LegacyEntityType.user].map(entity => (0, _bulk_delete_saved_objects.bulkDeleteSavedObjects)({
      deleteAll: true,
      savedObjectsClient: this.options.soClient,
      spaceId: namespace,
      savedObjectTemplate: `${entity}RiskScoreDashboards`
    }));
    await Promise.all(deleteDashboardsPromises);
    const newlegacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    return newlegacyRiskEngineStatus === _entity_analytics.RiskEngineStatusEnum.NOT_INSTALLED;
  }
  async getCurrentStatus() {
    const configuration = await this.getConfiguration();
    if (configuration) {
      return configuration.enabled ? _entity_analytics.RiskEngineStatusEnum.ENABLED : _entity_analytics.RiskEngineStatusEnum.DISABLED;
    }
    return _entity_analytics.RiskEngineStatusEnum.NOT_INSTALLED;
  }
  async getLegacyStatus({
    namespace
  }) {
    var _this$options$auditLo7;
    const transforms = await (0, _transforms.getLegacyTransforms)({
      namespace,
      esClient: this.options.esClient
    });
    (_this$options$auditLo7 = this.options.auditLogger) === null || _this$options$auditLo7 === void 0 ? void 0 : _this$options$auditLo7.log({
      message: 'System checked if the legacy risk engine is enabled',
      event: {
        action: _audit.RiskEngineAuditActions.RISK_ENGINE_GET_LEGACY_ENGINE_STATUS_GET,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.ACCESS,
        outcome: _audit2.AUDIT_OUTCOME.SUCCESS
      }
    });
    if (transforms.length === 0) {
      return _entity_analytics.RiskEngineStatusEnum.NOT_INSTALLED;
    }
    return _entity_analytics.RiskEngineStatusEnum.ENABLED;
  }
  async updateRiskEngineSavedObject(attributes) {
    try {
      const configuration = await this.getConfiguration();
      if (!configuration) {
        await (0, _saved_object_configuration.initSavedObjects)({
          savedObjectsClient: this.options.soClient,
          namespace: this.options.namespace
        });
      }
      return await (0, _saved_object_configuration.updateSavedObjectAttribute)({
        savedObjectsClient: this.options.soClient,
        attributes
      });
    } catch (e) {
      this.options.logger.error(`Error updating risk score engine saved object attributes: ${e.message}`);
      throw e;
    }
  }
}
exports.RiskEngineDataClient = RiskEngineDataClient;