"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskScoreCopyTimestampToEventIngested = exports.createMigrationTask = void 0;
var _risk_score_data_client = require("../risk_score/risk_score_data_client");
var _helpers = require("../risk_score/tasks/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'security-solution-ea-risk-score-copy-timestamp-to-event-ingested';
const TASK_ID = `${TASK_TYPE}-task-id`;
const TASK_TIMEOUT = '15m';
const TASK_SCOPE = ['securitySolution'];
const riskScoreCopyTimestampToEventIngested = async ({
  auditLogger,
  taskManager,
  logger,
  getStartServices
}) => {
  if (!taskManager) {
    return;
  }
  logger.debug(`Register task "${TASK_TYPE}"`);
  taskManager.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: `Copy Risk Score @timestamp value to events.ingested`,
      timeout: TASK_TIMEOUT,
      createTaskRunner: createMigrationTask({
        auditLogger,
        logger,
        getStartServices
      })
    }
  });
  const [_, depsStart] = await getStartServices();
  const taskManagerStart = depsStart.taskManager;
  if (taskManagerStart) {
    logger.debug(`Task scheduled: "${TASK_TYPE}"`);
    const now = new Date();
    try {
      await taskManagerStart.ensureScheduled({
        id: TASK_ID,
        taskType: TASK_TYPE,
        scheduledAt: now,
        runAt: now,
        scope: TASK_SCOPE,
        params: {},
        state: {}
      });
    } catch (e) {
      logger.error(`Error scheduling ${TASK_ID}, received ${e.message}`);
    }
  }
};
exports.riskScoreCopyTimestampToEventIngested = riskScoreCopyTimestampToEventIngested;
const createMigrationTask = ({
  getStartServices,
  logger,
  auditLogger
}) => () => {
  let abortController;
  return {
    run: async () => {
      var _riskScoreResponse$fa;
      abortController = new AbortController();
      const [coreStart] = await getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = (0, _helpers.buildScopedInternalSavedObjectsClientUnsafe)({
        coreStart,
        namespace: '*'
      });
      const riskScoreClient = new _risk_score_data_client.RiskScoreDataClient({
        esClient,
        logger,
        auditLogger,
        namespace: '*',
        soClient,
        kibanaVersion: '*'
      });
      const riskScoreResponse = await riskScoreClient.copyTimestampToEventIngestedForRiskScore(abortController.signal);
      const failures = (_riskScoreResponse$fa = riskScoreResponse.failures) === null || _riskScoreResponse$fa === void 0 ? void 0 : _riskScoreResponse$fa.map(failure => failure.cause);
      const hasFailures = failures && (failures === null || failures === void 0 ? void 0 : failures.length) > 0;
      logger.info(`Task "${TASK_TYPE}" finished. Updated documents: ${riskScoreResponse.updated}, failures: ${hasFailures ? failures.join('\n') : 0}`);
    },
    cancel: async () => {
      abortController.abort();
      logger.debug(`Task cancelled: "${TASK_TYPE}"`);
    }
  };
};
exports.createMigrationTask = createMigrationTask;