"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeBulkCreateAlerts = exports.bulkCreateSuppressedSequencesInMemory = exports.bulkCreateSuppressedAlertsInMemory = void 0;
var _constants = require("../constants");
var _utils = require("./utils");
var _constants2 = require("../../../../../common/detection_engine/constants");
var _partition_missing_fields_events = require("./partition_missing_fields_events");
var _rule_schema = require("../../../../../common/api/detection_engine/model/rule_schema");
var _bulk_create_with_suppression = require("./bulk_create_with_suppression");
var _robust_field_access = require("./source_fields_merging/utils/robust_field_access");
var _build_alert_group_from_sequence = require("../eql/build_alert_group_from_sequence");
var _factories = require("../factories");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * wraps, bulk create and suppress alerts in memory, also takes care of missing fields logic.
 * If parameter alertSuppression.missingFieldsStrategy configured not to be suppressed,
 * regular alerts will be created for such events without suppression
 */
const bulkCreateSuppressedAlertsInMemory = async ({
  sharedParams,
  enrichedEvents,
  toReturn,
  services,
  buildReasonMessage,
  alertSuppression,
  wrapSuppressedHits,
  mergeSourceAndFields = false,
  maxNumberOfAlertsMultiplier
}) => {
  var _alertSuppression$mis;
  const suppressOnMissingFields = ((_alertSuppression$mis = alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.missingFieldsStrategy) !== null && _alertSuppression$mis !== void 0 ? _alertSuppression$mis : _constants2.DEFAULT_SUPPRESSION_MISSING_FIELDS_STRATEGY) === _rule_schema.AlertSuppressionMissingFieldsStrategyEnum.suppress;
  let suppressibleEvents = enrichedEvents;
  let unsuppressibleWrappedDocs = [];
  if (!suppressOnMissingFields) {
    const partitionedEvents = (0, _partition_missing_fields_events.partitionMissingFieldsEvents)(enrichedEvents, (alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.groupBy) || [], ['fields'], mergeSourceAndFields);
    unsuppressibleWrappedDocs = (0, _factories.wrapHits)(sharedParams, partitionedEvents[1], buildReasonMessage);
    suppressibleEvents = partitionedEvents[0];
  }
  const suppressibleWrappedDocs = wrapSuppressedHits(suppressibleEvents, buildReasonMessage);
  return executeBulkCreateAlerts({
    sharedParams,
    suppressibleWrappedDocs,
    unsuppressibleWrappedDocs,
    toReturn,
    services,
    alertSuppression,
    maxNumberOfAlertsMultiplier
  });
};

/**
 * wraps, bulk create and suppress alerts in memory, also takes care of missing fields logic.
 * If parameter alertSuppression.missingFieldsStrategy configured not to be suppressed,
 * regular alerts will be created for such events without suppression
 */
exports.bulkCreateSuppressedAlertsInMemory = bulkCreateSuppressedAlertsInMemory;
const bulkCreateSuppressedSequencesInMemory = async ({
  sharedParams,
  sequences,
  toReturn,
  services,
  alertSuppression,
  buildReasonMessage
}) => {
  var _alertSuppression$mis2;
  const suppressOnMissingFields = ((_alertSuppression$mis2 = alertSuppression.missingFieldsStrategy) !== null && _alertSuppression$mis2 !== void 0 ? _alertSuppression$mis2 : _constants2.DEFAULT_SUPPRESSION_MISSING_FIELDS_STRATEGY) === _rule_schema.AlertSuppressionMissingFieldsStrategyEnum.suppress;
  const suppressibleWrappedSequences = [];
  const unsuppressibleWrappedDocs = [];
  sequences.forEach(sequence => {
    const alertGroupFromSequence = (0, _build_alert_group_from_sequence.buildAlertGroupFromSequence)({
      sharedParams,
      sequence,
      applyOverrides: true,
      buildReasonMessage
    });
    const shellAlert = alertGroupFromSequence.shellAlert;
    const buildingBlocks = alertGroupFromSequence.buildingBlocks;
    if (shellAlert) {
      if (!suppressOnMissingFields) {
        // does the shell alert have all the suppression fields?
        const hasEverySuppressionField = alertSuppression.groupBy.every(suppressionPath => (0, _robust_field_access.robustGet)({
          key: suppressionPath,
          document: shellAlert._source
        }) != null);
        if (!hasEverySuppressionField) {
          unsuppressibleWrappedDocs.push(shellAlert, ...buildingBlocks);
        } else {
          const wrappedWithSuppressionTerms = (0, _utils.buildShellAlertSuppressionTermsAndFields)({
            sharedParams,
            shellAlert,
            buildingBlockAlerts: buildingBlocks
          });
          suppressibleWrappedSequences.push(wrappedWithSuppressionTerms);
        }
      } else {
        const wrappedWithSuppressionTerms = (0, _utils.buildShellAlertSuppressionTermsAndFields)({
          sharedParams,
          shellAlert,
          buildingBlockAlerts: buildingBlocks
        });
        suppressibleWrappedSequences.push(wrappedWithSuppressionTerms);
      }
    }
  });
  return executeBulkCreateAlerts({
    sharedParams,
    suppressibleWrappedDocs: suppressibleWrappedSequences,
    unsuppressibleWrappedDocs,
    toReturn,
    services,
    alertSuppression
  });
};
exports.bulkCreateSuppressedSequencesInMemory = bulkCreateSuppressedSequencesInMemory;
/**
 * creates alerts in ES, both suppressed and unsuppressed
 */
const executeBulkCreateAlerts = async ({
  sharedParams,
  unsuppressibleWrappedDocs,
  suppressibleWrappedDocs,
  toReturn,
  services,
  alertSuppression,
  maxNumberOfAlertsMultiplier = _constants.MAX_SIGNALS_SUPPRESSION_MULTIPLIER
}) => {
  var _toReturn$suppressedA;
  const {
    tuple
  } = sharedParams;
  // max signals for suppression includes suppressed and created alerts
  // this allows to lift max signals limitation to higher value
  // and can detects events beyond default max_signals value
  const suppressionMaxSignals = maxNumberOfAlertsMultiplier * tuple.maxSignals;
  const suppressionDuration = alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.duration;
  const suppressionWindow = suppressionDuration ? `now-${suppressionDuration.value}${suppressionDuration.unit}` : tuple.from.toISOString();
  if (unsuppressibleWrappedDocs.length) {
    const unsuppressedResult = await (0, _factories.bulkCreate)({
      wrappedAlerts: unsuppressibleWrappedDocs,
      sharedParams,
      services,
      maxAlerts: tuple.maxSignals - toReturn.createdSignalsCount
    });
    (0, _utils.addToSearchAfterReturn)({
      current: toReturn,
      next: unsuppressedResult
    });
  }
  const bulkCreateResult = await (0, _bulk_create_with_suppression.bulkCreateWithSuppression)({
    sharedParams,
    wrappedDocs: suppressibleWrappedDocs,
    services,
    suppressionWindow,
    isSuppressionPerRuleExecution: !suppressionDuration,
    maxAlerts: tuple.maxSignals - toReturn.createdSignalsCount
  });
  (0, _utils.addToSearchAfterReturn)({
    current: toReturn,
    next: bulkCreateResult
  });
  const alertsWereTruncated = ((_toReturn$suppressedA = toReturn.suppressedAlertsCount) !== null && _toReturn$suppressedA !== void 0 ? _toReturn$suppressedA : 0) + toReturn.createdSignalsCount >= suppressionMaxSignals || toReturn.createdSignalsCount >= tuple.maxSignals;
  return {
    ...bulkCreateResult,
    alertsWereTruncated
  };
};
exports.executeBulkCreateAlerts = executeBulkCreateAlerts;