"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformHitToAlert = void 0;
var _lodash = require("lodash");
var _zodHelpers = require("@kbn/zod-helpers");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _strategies = require("../../utils/source_fields_merging/strategies");
var _build_alert = require("./build_alert");
var _build_rule_name_from_mapping = require("../../utils/mappings/build_rule_name_from_mapping");
var _build_severity_from_mapping = require("../../utils/mappings/build_severity_from_mapping");
var _build_risk_score_from_mapping = require("../../utils/mappings/build_risk_score_from_mapping");
var _traverse_and_mutate_doc = require("./traverse_and_mutate_doc");
var _field_names = require("../../../../../../common/field_maps/field_names");
var _robust_field_access = require("../../utils/source_fields_merging/utils/robust_field_access");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isSourceDoc = hit => {
  return hit._source != null && hit._id != null;
};
/**
 * Formats the search_after result for insertion into the signals index. We first create a
 * "best effort" merged "fields" with the "_source" object, then build the signal object,
 * then the event object, and finally we strip away any additional temporary data that was added
 * such as the "threshold_result".
 * @param completeRule The rule saved object to build overrides
 * @param doc The SignalSourceHit with "_source", "fields", and additional data such as "threshold_result"
 * @returns The body that can be added to a bulk call for inserting the signal.
 */
const transformHitToAlert = ({
  sharedParams,
  doc,
  applyOverrides,
  buildReasonMessage,
  alertUuid
}) => {
  var _mergedDoc$_source;
  const mergedDoc = (0, _strategies.getMergeStrategy)(sharedParams.mergeStrategy)({
    doc,
    ignoreFields: sharedParams.ignoreFields,
    ignoreFieldsRegexes: sharedParams.ignoreFieldsRegexes
  });
  const thresholdResult = (_mergedDoc$_source = mergedDoc._source) === null || _mergedDoc$_source === void 0 ? void 0 : _mergedDoc$_source.threshold_result;
  const {
    completeRule,
    spaceId,
    inputIndex: indicesToQuery,
    publicBaseUrl,
    alertTimestampOverride,
    intendedTimestamp,
    ruleExecutionLogger
  } = sharedParams;
  if (isSourceDoc(mergedDoc)) {
    var _mergedDoc$_source2, _mergedDoc$_source3, _mergedDoc$_source4, _overrides$nameOverri, _overrides$severityOv;
    const overrides = applyOverrides ? {
      nameOverride: (0, _build_rule_name_from_mapping.buildRuleNameFromMapping)({
        eventSource: (_mergedDoc$_source2 = mergedDoc._source) !== null && _mergedDoc$_source2 !== void 0 ? _mergedDoc$_source2 : {},
        ruleName: completeRule.ruleConfig.name,
        ruleNameMapping: completeRule.ruleParams.ruleNameOverride
      }).ruleName,
      severityOverride: (0, _build_severity_from_mapping.buildSeverityFromMapping)({
        eventSource: (_mergedDoc$_source3 = mergedDoc._source) !== null && _mergedDoc$_source3 !== void 0 ? _mergedDoc$_source3 : {},
        severity: completeRule.ruleParams.severity,
        severityMapping: completeRule.ruleParams.severityMapping
      }).severity,
      riskScoreOverride: (0, _build_risk_score_from_mapping.buildRiskScoreFromMapping)({
        eventSource: (_mergedDoc$_source4 = mergedDoc._source) !== null && _mergedDoc$_source4 !== void 0 ? _mergedDoc$_source4 : {},
        riskScore: completeRule.ruleParams.riskScore,
        riskScoreMapping: (0, _zodHelpers.requiredOptional)(completeRule.ruleParams.riskScoreMapping)
      }).riskScore
    } : undefined;
    const reason = buildReasonMessage({
      name: (_overrides$nameOverri = overrides === null || overrides === void 0 ? void 0 : overrides.nameOverride) !== null && _overrides$nameOverri !== void 0 ? _overrides$nameOverri : completeRule.ruleConfig.name,
      severity: (_overrides$severityOv = overrides === null || overrides === void 0 ? void 0 : overrides.severityOverride) !== null && _overrides$severityOv !== void 0 ? _overrides$severityOv : completeRule.ruleParams.severity,
      mergedDoc
    });
    const alertFields = (0, _build_alert.buildAlertFields)({
      docs: [mergedDoc],
      completeRule,
      spaceId,
      reason,
      indicesToQuery,
      alertUuid,
      publicBaseUrl,
      alertTimestampOverride,
      overrides,
      intendedTimestamp
    });
    const {
      result: validatedSource,
      removed: removedSourceFields
    } = (0, _traverse_and_mutate_doc.traverseAndMutateDoc)(mergedDoc._source);

    // The `alertFields` we add to alerts contain `event.kind: 'signal'` in dot notation. To avoid duplicating `event.kind`,
    // we remove any existing `event.kind` field here before we merge `alertFields` into `validatedSource` later on
    if ((0, _robust_field_access.robustGet)({
      key: _ruleDataUtils.EVENT_KIND,
      document: validatedSource
    }) != null) {
      (0, _robust_field_access.robustSet)({
        key: _ruleDataUtils.EVENT_KIND,
        document: validatedSource,
        valueToSet: undefined
      });
    }
    if (removedSourceFields.length) {
      ruleExecutionLogger === null || ruleExecutionLogger === void 0 ? void 0 : ruleExecutionLogger.debug('Following fields were removed from alert source as ECS non-compliant:', JSON.stringify(removedSourceFields));
    }
    (0, _lodash.merge)(validatedSource, alertFields);
    if (thresholdResult != null && (0, _build_alert.isThresholdResult)(thresholdResult)) {
      validatedSource[_field_names.ALERT_THRESHOLD_RESULT] = thresholdResult;
    }
    return validatedSource;
  }
  throw Error('Error building alert from source document.');
};
exports.transformHitToAlert = transformHitToAlert;