"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.upgradePrebuiltRules = void 0;
var _constants = require("../../../../../../common/constants");
var _promise_pool = require("../../../../../utils/promise_pool");
var _with_security_span = require("../../../../../utils/with_security_span");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Upgrades existing prebuilt rules given a set of rules and output index.
 * This implements a chunked approach to not saturate network connections and
 * avoid being a "noisy neighbor".
 * @param detectionRulesClient IDetectionRulesClient
 * @param rules The rules to apply the update for
 * @param logger Logger to log debug messages
 */
const upgradePrebuiltRules = async (detectionRulesClient, rules, logger) => (0, _with_security_span.withSecuritySpan)('upgradePrebuiltRules', async () => {
  logger.debug(`upgradePrebuiltRules: Upgrading prebuilt rules - started. Rules to upgrade: ${rules.length}`);
  const result = await (0, _promise_pool.initPromisePool)({
    concurrency: _constants.MAX_RULES_TO_UPDATE_IN_PARALLEL,
    items: rules,
    executor: async rule => {
      return detectionRulesClient.upgradePrebuiltRule({
        ruleAsset: rule
      });
    }
  });
  logger.debug(`upgradePrebuiltRules: Upgrading prebuilt rules - done. Upgraded: ${result.results.length}. Failed to upgrade: ${result.errors.length}.`);
  return result;
});
exports.upgradePrebuiltRules = upgradePrebuiltRules;