"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.securityWorkflowInsightsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _errors = require("./errors");
var _helpers = require("./helpers");
var _constants = require("./constants");
var _builders = require("./builders");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_PAGE_SIZE = 10;
class SecurityWorkflowInsightsService {
  constructor() {
    (0, _defineProperty2.default)(this, "setup$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "start$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "ds", void 0);
    (0, _defineProperty2.default)(this, "_esClient", void 0);
    (0, _defineProperty2.default)(this, "_endpointContext", void 0);
    (0, _defineProperty2.default)(this, "_logger", void 0);
    (0, _defineProperty2.default)(this, "_isInitialized", (0, _rxjs.firstValueFrom)((0, _rxjs.combineLatest)([this.setup$, this.start$])));
    (0, _defineProperty2.default)(this, "isFeatureEnabled", false);
  }
  get isInitialized() {
    return this._isInitialized;
  }
  setup({
    kibanaVersion,
    logger,
    isFeatureEnabled,
    endpointContext
  }) {
    this.isFeatureEnabled = isFeatureEnabled;
    if (!isFeatureEnabled) {
      return;
    }
    this._logger = logger;
    this._endpointContext = endpointContext;
    try {
      this.ds = (0, _helpers.createDatastream)(kibanaVersion);
    } catch (err) {
      this.logger.warn(new _errors.SecurityWorkflowInsightsFailedInitialized(err.message).message);
      return;
    }
    this.setup$.next();
  }
  async start({
    esClient
  }) {
    if (!this.isFeatureEnabled) {
      return;
    }
    this._esClient = esClient;
    await (0, _rxjs.firstValueFrom)(this.setup$);
    try {
      var _this$ds;
      await (0, _helpers.createPipeline)(esClient);
      await ((_this$ds = this.ds) === null || _this$ds === void 0 ? void 0 : _this$ds.install({
        logger: this.logger,
        esClient,
        pluginStop$: this.stop$
      }));
      try {
        await esClient.indices.createDataStream({
          name: _constants.DATA_STREAM_NAME
        });
      } catch (err) {
        var _err$body, _err$body$error;
        if ((err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'resource_already_exists_exception') {
          this.logger.debug(`Datastream ${_constants.DATA_STREAM_NAME} already exists`);
        } else {
          throw new _errors.SecurityWorkflowInsightsFailedInitialized(err.message);
        }
      }
    } catch (err) {
      this.logger.warn(err.message);
      return;
    }
    this.start$.next();
  }
  stop() {
    this.setup$.next();
    this.setup$.complete();
    this.start$.next();
    this.start$.complete();
    this.stop$.next();
    this.stop$.complete();
  }
  async createFromDefendInsights(defendInsights, request) {
    await this.isInitialized;
    const workflowInsights = await (0, _builders.buildWorkflowInsights)({
      defendInsights,
      request,
      endpointMetadataService: this.endpointContext.getEndpointMetadataService(),
      esClient: this.esClient
    });
    const uniqueInsights = (0, _helpers.getUniqueInsights)(workflowInsights);
    return Promise.all(uniqueInsights.map(insight => this.create(insight)));
  }
  async create(insight) {
    await this.isInitialized;
    const id = (0, _helpers.generateInsightId)(insight);
    const remediationExists = await (0, _helpers.checkIfRemediationExists)({
      insight,
      exceptionListsClient: this.endpointContext.getExceptionListsClient()
    });
    if (remediationExists) {
      return;
    }

    // if insight already exists, update instead
    const existingInsights = await this.fetch({
      ids: [id]
    });
    if (existingInsights.length) {
      return this.update(id, insight, existingInsights[0]._index);
    }
    const response = await this.esClient.index({
      index: _constants.DATA_STREAM_NAME,
      id,
      body: insight,
      refresh: 'wait_for',
      op_type: 'create'
    });
    return response;
  }
  async update(id, insight, backingIndex) {
    await this.isInitialized;
    let index = backingIndex;
    if (!index) {
      const retrievedInsight = (await this.fetch({
        ids: [id]
      }))[0];
      index = retrievedInsight === null || retrievedInsight === void 0 ? void 0 : retrievedInsight._index;
    }
    if (!index) {
      throw new Error('invalid backing index for updating workflow insight');
    }
    const response = await this.esClient.update({
      index,
      id,
      body: {
        doc: insight
      },
      refresh: 'wait_for'
    });
    return response;
  }
  async fetch(params) {
    var _params$size, _params$from, _response$hits$hits, _response$hits;
    await this.isInitialized;
    const size = (_params$size = params === null || params === void 0 ? void 0 : params.size) !== null && _params$size !== void 0 ? _params$size : DEFAULT_PAGE_SIZE;
    const from = (_params$from = params === null || params === void 0 ? void 0 : params.from) !== null && _params$from !== void 0 ? _params$from : 0;
    const termFilters = params ? (0, _helpers.buildEsQueryParams)(params) : [];
    const response = await this.esClient.search({
      index: _constants.DATA_STREAM_NAME,
      body: {
        query: {
          bool: {
            must: termFilters
          }
        },
        size,
        from
      }
    });
    return (_response$hits$hits = response === null || response === void 0 ? void 0 : (_response$hits = response.hits) === null || _response$hits === void 0 ? void 0 : _response$hits.hits) !== null && _response$hits$hits !== void 0 ? _response$hits$hits : [];
  }
  get esClient() {
    if (!this._esClient) {
      throw new _errors.SecurityWorkflowInsightsFailedInitialized('no elasticsearch client found');
    }
    return this._esClient;
  }
  get logger() {
    if (!this._logger) {
      throw new _errors.SecurityWorkflowInsightsFailedInitialized('no logger found');
    }
    return this._logger;
  }
  get endpointContext() {
    if (!this._endpointContext) {
      throw new _errors.SecurityWorkflowInsightsFailedInitialized('no endpoint context found');
    }
    return this._endpointContext;
  }
}
const securityWorkflowInsightsService = exports.securityWorkflowInsightsService = new SecurityWorkflowInsightsService();