"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerActionFileInfoRoute = exports.getActionFileInfoRouteHandler = void 0;
var _utils = require("./utils");
var _stringify = require("../../utils/stringify");
var _endpoint = require("../../../../common/api/endpoint");
var _services = require("../../services");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _error_handler = require("../error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getActionFileInfoRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('actionFileInfo');
  return async (context, req, res) => {
    logger.debug(() => `Get response action file info:\n${(0, _stringify.stringify)(req.params)}`);
    const {
      action_id: requestActionId,
      file_id: fileId
    } = req.params;
    const coreContext = await context.core;
    try {
      const esClient = coreContext.elasticsearch.client.asInternalUser;
      const {
        agentType
      } = await (0, _services.getActionAgentType)(esClient, requestActionId);
      const user = coreContext.security.authc.getCurrentUser();
      const casesClient = await endpointContext.service.getCasesClient(req);
      const connectorActions = (await context.actions).getActionsClient();
      const responseActionsClient = (0, _services.getResponseActionsClient)(agentType, {
        esClient,
        casesClient,
        endpointService: endpointContext.service,
        username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
        connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
      });
      const response = {
        data: await responseActionsClient.getFileInfo(requestActionId, fileId)
      };
      return res.ok({
        body: response
      });
    } catch (error) {
      return (0, _error_handler.errorHandler)(logger, res, error);
    }
  };
};
exports.getActionFileInfoRouteHandler = getActionFileInfoRouteHandler;
const registerActionFileInfoRoute = (router, endpointContext) => {
  router.versioned.get({
    access: 'public',
    path: _constants.ACTION_AGENT_FILE_INFO_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionFileInfoSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    any: ['canWriteFileOperations', 'canWriteExecuteOperations', 'canGetRunningProcesses']
  }, endpointContext.logFactory.get('actionFileInfo'), getActionFileInfoRouteHandler(endpointContext), _utils.ensureUserHasAuthzToFilesForAction));
};
exports.registerActionFileInfoRoute = registerActionFileInfoRoute;