"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRODUCT_DOCUMENTATION_TOOL = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _inferenceCommon = require("@kbn/inference-common");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolDetails = {
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: 'Use this tool to retrieve documentation about Elastic products. You can retrieve documentation about the Elastic stack, such as Kibana and Elasticsearch, or for Elastic solutions, such as Elastic Security, Elastic Observability or Elastic Enterprise Search.',
  id: 'product-documentation-tool',
  name: 'ProductDocumentationTool'
};
const PRODUCT_DOCUMENTATION_TOOL = exports.PRODUCT_DOCUMENTATION_TOOL = {
  ...toolDetails,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    return params.llmTasks != null && params.connectorId != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      connectorId,
      llmTasks,
      request,
      contentReferencesStore
    } = params;
    return (0, _tools.tool)(async ({
      query,
      product
    }) => {
      const response = await llmTasks.retrieveDocumentation({
        searchTerm: query,
        products: product ? [product] : undefined,
        max: 3,
        connectorId,
        request,
        functionCalling: 'auto',
        inferenceId: _inferenceCommon.defaultInferenceEndpoints.ELSER
      });
      const enrichedDocuments = response.documents.map(enrichDocument(contentReferencesStore));
      return {
        content: {
          documents: enrichedDocuments
        }
      };
    }, {
      name: toolDetails.name,
      description: params.description || toolDetails.description,
      schema: _zod.z.object({
        query: _zod.z.string().describe(`The query to use to retrieve documentation
            Examples:
            - "How to enable TLS for Elasticsearch?"
            - "What is Kibana Security?"`),
        product: _zod.z.enum(['kibana', 'elasticsearch', 'observability', 'security']).describe(`If specified, will filter the products to retrieve documentation for
            Possible options are:
            - "kibana": Kibana product
            - "elasticsearch": Elasticsearch product
            - "observability": Elastic Observability solution
            - "security": Elastic Security solution
            If not specified, will search against all products
            `).optional()
      }),
      tags: ['product-documentation']
    });
  }
};
const enrichDocument = contentReferencesStore => {
  return document => {
    const reference = contentReferencesStore.add(p => (0, _elasticAssistantCommon.productDocumentationReference)(p.id, document.title, document.url));
    return {
      ...document,
      citation: (0, _elasticAssistantCommon.contentReferenceBlock)(reference)
    };
  };
};