"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getShortlistIndexPatterns = void 0;
var _messages = require("@langchain/core/messages");
var _langgraph = require("@langchain/langgraph");
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ShortlistedIndexPatterns = _zod.z.object({
  shortlistedIndexPatterns: _zod.z.array(_zod.z.string()).describe('Shortlisted index patterns')
}).describe('Object containing array of shortlisted index patterns that might be used to generate the query');
const getShortlistIndexPatterns = async ({
  createLlmInstance
}) => {
  const llm = await createLlmInstance();
  return async state => {
    var _state$input;
    const systemMessage = new _messages.SystemMessage({
      content: `You are a security analyst who is an expert in Elasticsearch and particularly writing Elastic Search queries. You have been given a list of index patterns and an explanation of the query we would like to generate. 
To generate the query you first need to identify which index pattern should be used. To do this you short list a maximum of 3 index patterns that are the most likely to contain the fields required to write the query. If the query explanation explicitly states which index to use, then only shortlist the matching index pattern. Select a variety of index patterns and sort them by highest relevance first.`
    });
    const humanMessage = new _messages.HumanMessage({
      content: `Available index patterns:\n ${state.indexPatterns.join('\n')} \n\n Explanation of the query: \n\n ${(_state$input = state.input) === null || _state$input === void 0 ? void 0 : _state$input.question} \n\n Based on this information, please shortlist a maximum of 3 index patterns that are the most likely to contain the fields required to write the query.`
    });
    try {
      const result = await llm.withStructuredOutput(ShortlistedIndexPatterns, {
        name: 'shortlistedIndexPatterns'
      }).withRetry({
        stopAfterAttempt: 3
      }).invoke([systemMessage, humanMessage]);
      return new _langgraph.Command({
        update: {
          shortlistedIndexPatterns: result.shortlistedIndexPatterns.slice(0, 3)
        }
      });
    } catch (error) {
      return new _langgraph.Command({});
    }
  };
};
exports.getShortlistIndexPatterns = getShortlistIndexPatterns;