"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexFleetServerAgent = exports.deleteIndexedFleetAgents = exports.buildFleetAgentBulkCreateOperations = void 0;
var _common = require("@kbn/fleet-plugin/common");
var _common2 = require("@kbn/spaces-plugin/common");
var _fleet_agent_generator = require("../data_generators/fleet_agent_generator");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultFleetAgentGenerator = new _fleet_agent_generator.FleetAgentGenerator();
const generateFleetAgentEsHitForEndpointHost = (endpointHost, agentPolicyId, kibanaVersion = '8.0.0', fleetAgentGenerator = defaultFleetAgentGenerator, spaceId = _common2.DEFAULT_SPACE_ID) => {
  return fleetAgentGenerator.generateEsHit({
    _id: endpointHost.agent.id,
    _source: {
      agent: {
        id: endpointHost.agent.id,
        version: endpointHost.agent.version
      },
      local_metadata: {
        elastic: {
          agent: {
            id: endpointHost.agent.id,
            version: kibanaVersion
          }
        },
        host: {
          ...endpointHost.host
        },
        os: {
          ...endpointHost.host.os
        }
      },
      policy_id: agentPolicyId,
      namespaces: [spaceId]
    }
  });
};
/**
 * Creates an array of ES records with Fleet Agents that are associated with the provided set of Endpoint Agents.
 * Array can be used with the `bulk()` API's `operations` option.
 * @param endpoints
 * @param agentPolicyId
 * @param kibanaVersion
 * @param fleetAgentGenerator
 */
const buildFleetAgentBulkCreateOperations = ({
  endpoints,
  agentPolicyId,
  kibanaVersion = '8.0.0',
  fleetAgentGenerator = defaultFleetAgentGenerator,
  spaceId = _common2.DEFAULT_SPACE_ID
}) => {
  const response = {
    operations: [],
    agents: [],
    fleetAgentsIndex: _common.AGENTS_INDEX
  };
  for (const endpointHost of endpoints) {
    const agentDoc = generateFleetAgentEsHitForEndpointHost(endpointHost, agentPolicyId, kibanaVersion, fleetAgentGenerator, spaceId);
    response.operations.push({
      create: {
        _index: agentDoc._index,
        _id: agentDoc._id
      }
    }, agentDoc._source);
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    response.agents.push(agentDoc._source);
  }
  return response;
};
exports.buildFleetAgentBulkCreateOperations = buildFleetAgentBulkCreateOperations;
const deleteIndexedFleetAgents = async (esClient, indexedData) => {
  const response = {
    agents: undefined
  };
  if (indexedData.agents.length) {
    response.agents = await esClient.deleteByQuery({
      index: `${indexedData.fleetAgentsIndex}-*`,
      wait_for_completion: true,
      body: {
        query: {
          bool: {
            filter: [{
              terms: {
                'local_metadata.elastic.agent.id': indexedData.agents.map(agent => agent.local_metadata.elastic.agent.id)
              }
            }]
          }
        }
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  return response;
};
exports.deleteIndexedFleetAgents = deleteIndexedFleetAgents;
const indexFleetServerAgent = async (esClient, log = (0, _utils.createToolingLogger)(), overrides = {}) => {
  const doc = defaultFleetAgentGenerator.generateEsHit({
    _source: overrides
  });
  const indexRequest = {
    index: doc._index,
    id: doc._id,
    body: doc._source,
    op_type: 'create',
    refresh: 'wait_for'
  };
  log.verbose(`Indexing new fleet agent with:\n${JSON.stringify(indexRequest, null, 2)}`);
  await esClient.index(indexRequest).catch(_utils.wrapErrorAndRejectPromise);
  return {
    fleetAgentsIndex: doc._index,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    agents: [doc._source]
  };
};
exports.indexFleetServerAgent = indexFleetServerAgent;