"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexEndpointHostDocs = exports.deleteIndexedEndpointHosts = void 0;
var _lodash = require("lodash");
var _uuid = require("uuid");
var _errors = require("../errors");
var _usage_tracker = require("./usage_tracker");
var _generate_data = require("../generate_data");
var _index_fleet_agent = require("./index_fleet_agent");
var _index_endpoint_fleet_actions = require("./index_endpoint_fleet_actions");
var _index_fleet_endpoint_policy = require("./index_fleet_endpoint_policy");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Indexes the requested number of documents for the endpoint host metadata currently being output by the generator.
 * Endpoint Host metadata documents are added to an index that is set as "append only", thus one Endpoint host could
 * have multiple documents in that index.
 *
 * @param numDocs
 * @param client
 * @param kbnClient
 * @param realPolicies
 * @param epmEndpointPackage
 * @param metadataIndex
 * @param policyResponseIndex
 * @param enrollFleet
 * @param generator
 * @param disableEndpointActionsForHost
 */
const indexEndpointHostDocs = exports.indexEndpointHostDocs = _usage_tracker.usageTracker.track('indexEndpointHostDocs', async ({
  numDocs,
  client,
  kbnClient,
  realPolicies,
  epmEndpointPackage,
  metadataIndex,
  policyResponseIndex,
  enrollFleet,
  generator,
  withResponseActions = true,
  numResponseActions = 1,
  alertIds
}) => {
  const timeBetweenDocs = 6 * 3600 * 1000; // 6 hours between metadata documents
  const timestamp = new Date().getTime();
  const kibanaVersion = await fetchKibanaVersion(kbnClient);
  const activeSpaceId = await (0, _utils.fetchActiveSpaceId)(kbnClient);
  const response = {
    hosts: [],
    agents: [],
    policyResponses: [],
    metadataIndex,
    policyResponseIndex,
    fleetAgentsIndex: '',
    endpointActionResponses: [],
    endpointActionResponsesIndex: '',
    endpointActions: [],
    endpointActionsIndex: '',
    actionResponses: [],
    responsesIndex: '',
    actions: [],
    actionsIndex: '',
    integrationPolicies: [],
    agentPolicies: []
  };
  let hostMetadata;
  let wasAgentEnrolled = false;
  const bulkOperations = [];
  for (let j = 0; j < numDocs; j++) {
    generator.updateHostData();
    generator.updateHostPolicyData({
      excludeInitialPolicy: true
    });
    hostMetadata = generator.generateHostMetadata(timestamp - timeBetweenDocs * (numDocs - j - 1), _generate_data.EndpointDocGenerator.createDataStreamFromIndex(metadataIndex));
    let agentId = hostMetadata.agent.id;
    if (enrollFleet) {
      const {
        id: appliedPolicyId,
        name: appliedPolicyName
      } = hostMetadata.Endpoint.policy.applied;
      const uniqueAppliedPolicyName = `${appliedPolicyName}-${(0, _uuid.v4)()}`;

      // If we don't yet have a "real" policy record, then create it now in ingest (package config)
      if (!realPolicies[appliedPolicyId]) {
        const createdPolicies = await (0, _index_fleet_endpoint_policy.indexFleetEndpointPolicy)(kbnClient, uniqueAppliedPolicyName, epmEndpointPackage.version);
        (0, _utils.mergeAndAppendArrays)(response, createdPolicies);

        // eslint-disable-next-line require-atomic-updates
        realPolicies[appliedPolicyId] = createdPolicies.integrationPolicies[0];
      }

      // If we did not yet enroll an agent for this Host, do it now that we have good policy id
      if (!wasAgentEnrolled) {
        var _agentOperations$agen, _agentOperations$agen2, _agentOperations$agen3;
        wasAgentEnrolled = true;
        const agentOperations = {
          agents: [],
          fleetAgentsIndex: '',
          operations: []
        };
        realPolicies[appliedPolicyId].policy_ids.forEach(policyId => {
          const {
            agents,
            fleetAgentsIndex,
            operations
          } = (0, _index_fleet_agent.buildFleetAgentBulkCreateOperations)({
            endpoints: [hostMetadata],
            agentPolicyId: policyId,
            spaceId: activeSpaceId,
            kibanaVersion
          });
          agentOperations.agents = [...agentOperations.agents, ...agents];
          agentOperations.fleetAgentsIndex = fleetAgentsIndex;
          agentOperations.operations = [...agentOperations.operations, ...operations];
        });
        bulkOperations.push(...agentOperations.operations);
        agentId = (_agentOperations$agen = (_agentOperations$agen2 = agentOperations.agents[0]) === null || _agentOperations$agen2 === void 0 ? void 0 : (_agentOperations$agen3 = _agentOperations$agen2.agent) === null || _agentOperations$agen3 === void 0 ? void 0 : _agentOperations$agen3.id) !== null && _agentOperations$agen !== void 0 ? _agentOperations$agen : agentId;
        (0, _utils.mergeAndAppendArrays)(response, {
          agents: agentOperations.agents,
          fleetAgentsIndex: agentOperations.fleetAgentsIndex
        });
      }

      // Update the Host metadata record with the ID of the "real" policy along with the enrolled agent id
      hostMetadata = {
        ...hostMetadata,
        agent: {
          ...hostMetadata.agent,
          id: agentId
        },
        elastic: {
          ...hostMetadata.elastic,
          agent: {
            ...hostMetadata.elastic.agent,
            id: agentId
          }
        },
        Endpoint: {
          ...hostMetadata.Endpoint,
          policy: {
            ...hostMetadata.Endpoint.policy,
            applied: {
              ...hostMetadata.Endpoint.policy.applied,
              id: realPolicies[appliedPolicyId].id
            }
          }
        }
      };

      // Create some fleet endpoint actions and .logs-endpoint actions for this Host
      if (withResponseActions) {
        // `count` logic matches that of `indexEndpointAndFleetActionsForHost()`. Unclear why the number of
        // actions to create will be 5 more than the amount requested if that amount was grater than 1
        const count = numResponseActions === 1 ? numResponseActions : generator.randomN(5) + numResponseActions;
        const {
          operations,
          ...indexFleetActions
        } = (0, _index_endpoint_fleet_actions.buildIEndpointAndFleetActionsBulkOperations)({
          endpoints: [hostMetadata],
          count,
          alertIds
        });
        bulkOperations.push(...operations);
        (0, _utils.mergeAndAppendArrays)(response, indexFleetActions);
      }
    }
    bulkOperations.push({
      create: {
        _index: metadataIndex
      }
    }, hostMetadata);
    const hostPolicyResponse = generator.generatePolicyResponse({
      ts: timestamp - timeBetweenDocs * (numDocs - j - 1),
      policyDataStream: _generate_data.EndpointDocGenerator.createDataStreamFromIndex(policyResponseIndex)
    });
    bulkOperations.push({
      create: {
        _index: policyResponseIndex
      }
    }, hostPolicyResponse);

    // Clone the hostMetadata and policyResponse document to ensure that no shared state
    // (as a result of using the generator) is returned across docs.
    response.hosts.push((0, _lodash.cloneDeep)(hostMetadata));
    response.policyResponses.push((0, _lodash.cloneDeep)(hostPolicyResponse));
  }
  const bulkResponse = await client.bulk({
    operations: bulkOperations,
    refresh: 'wait_for'
  }, {
    headers: {
      'X-elastic-product-origin': 'fleet'
    }
  }).catch(_utils.wrapErrorAndRejectPromise);
  if (bulkResponse.errors) {
    throw new _errors.EndpointError(`indexEndpointHostDocs(): ES Bulk action failed\n\n${JSON.stringify(bulkResponse, null, 2)}`, bulkResponse);
  }
  return response;
});
const fetchKibanaVersion = async kbnClient => {
  const version = (await kbnClient.request({
    path: '/api/status',
    method: 'GET'
  })).data.version.number;
  if (!version) {
    throw new _utils.EndpointDataLoadingError('failed to get kibana version via `/api/status` api');
  }
  return version;
};
const deleteIndexedEndpointHosts = async (esClient, kbnClient, indexedData) => {
  const response = {
    hosts: undefined,
    policyResponses: undefined,
    agents: undefined,
    responses: undefined,
    actions: undefined,
    endpointActionRequests: undefined,
    endpointActionResponses: undefined,
    integrationPolicies: undefined,
    agentPolicies: undefined
  };
  if (indexedData.hosts.length) {
    const query = {
      bool: {
        filter: [{
          terms: {
            'agent.id': indexedData.hosts.map(host => host.agent.id)
          }
        }]
      }
    };
    response.hosts = await esClient.deleteByQuery({
      index: indexedData.metadataIndex,
      wait_for_completion: true,
      query
    }).catch(_utils.wrapErrorAndRejectPromise);

    // Delete from the transform destination index
    await esClient.deleteByQuery({
      index: _constants.metadataCurrentIndexPattern,
      wait_for_completion: true,
      query
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  if (indexedData.policyResponses.length) {
    response.policyResponses = await esClient.deleteByQuery({
      index: indexedData.policyResponseIndex,
      wait_for_completion: true,
      query: {
        bool: {
          filter: [{
            terms: {
              'agent.id': indexedData.policyResponses.map(policyResponse => policyResponse.agent.id)
            }
          }]
        }
      }
    }).catch(_utils.wrapErrorAndRejectPromise);
  }
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_fleet_agent.deleteIndexedFleetAgents)(esClient, indexedData));
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_endpoint_fleet_actions.deleteIndexedEndpointAndFleetActions)(esClient, indexedData));
  (0, _utils.mergeAndAppendArrays)(response, await (0, _index_fleet_endpoint_policy.deleteIndexedFleetEndpointPolicies)(kbnClient, indexedData));
  return response;
};
exports.deleteIndexedEndpointHosts = deleteIndexedEndpointHosts;