"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRequestProfile = void 0;
var _i18n = require("@kbn/i18n");
var _app_context = require("../contexts/app_context");
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const extractProfilerErrorMessage = e => {
  var _e$body, _e$body$attributes, _e$body$attributes$er, _e$body2;
  if ((_e$body = e.body) !== null && _e$body !== void 0 && (_e$body$attributes = _e$body.attributes) !== null && _e$body$attributes !== void 0 && (_e$body$attributes$er = _e$body$attributes.error) !== null && _e$body$attributes$er !== void 0 && _e$body$attributes$er.reason) {
    const {
      reason,
      line,
      col
    } = e.body.attributes.error;
    if (typeof line === 'number' && typeof col === 'number') {
      return `${reason} at line: ${line - 1} col: ${col}`;
    }
  }
  if ((_e$body2 = e.body) !== null && _e$body2 !== void 0 && _e$body2.message) {
    return e.body.message;
  }
  return;
};
const useRequestProfile = () => {
  const {
    http,
    notifications,
    getLicenseStatus
  } = (0, _app_context.useAppContext)();
  const licenseEnabled = getLicenseStatus().valid;
  return async ({
    query,
    index
  }) => {
    if (!licenseEnabled) {
      return {
        data: null
      };
    }
    const {
      error,
      parsed
    } = (0, _lib.checkForParseErrors)(query);
    if (error) {
      var _error$body;
      notifications.addDanger({
        'data-test-subj': 'jsonParseErrorToast',
        title: _i18n.i18n.translate('xpack.searchProfiler.errorToastTitle', {
          defaultMessage: 'JSON parse error'
        }),
        text: ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.message) || error.message
      });
      return {
        data: null
      };
    }
    // Shortcut the network request if we have json with shards already...
    if (parsed.profile && parsed.profile.shards) {
      return {
        data: parsed.profile.shards
      };
    }
    const payload = {
      query: parsed
    };
    if (index == null || index === '') {
      payload.index = '_all';
    } else {
      payload.index = index;
    }
    try {
      const resp = await http.post('../api/searchprofiler/profile', {
        body: JSON.stringify(payload),
        headers: {
          'Content-Type': 'application/json'
        }
      });
      if (!resp.ok) {
        return {
          data: null,
          error: resp.err.msg
        };
      }

      // If a user attempts to run Search Profiler without any indices,
      // _shards=0 and a "profile" output will not be returned
      if (resp.resp._shards.total === 0) {
        notifications.addDanger({
          'data-test-subj': 'noShardsNotification',
          title: _i18n.i18n.translate('xpack.searchProfiler.errorNoShardsTitle', {
            defaultMessage: 'Unable to profile'
          }),
          text: _i18n.i18n.translate('xpack.searchProfiler.errorNoShardsDescription', {
            defaultMessage: 'Verify your index input matches a valid index'
          })
        });
        return {
          data: null
        };
      }
      return {
        data: resp.resp.profile.shards
      };
    } catch (e) {
      const profilerErrorMessage = extractProfilerErrorMessage(e);
      if (profilerErrorMessage) {
        notifications.addError(e, {
          title: e.message,
          toastMessage: profilerErrorMessage
        });
      } else {
        // Otherwise just report the original error
        notifications.addError(e, {
          title: _i18n.i18n.translate('xpack.searchProfiler.errorSomethingWentWrongTitle', {
            defaultMessage: 'Something went wrong'
          })
        });
      }
      return {
        data: null
      };
    }
  };
};
exports.useRequestProfile = useRequestProfile;