"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRetriever = createRetriever;
exports.defineRoutes = defineRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _common = require("../common");
var _events = require("./analytics/events");
var _fetch_query_source_fields = require("./lib/fetch_query_source_fields");
var _assist = require("./utils/assist");
var _conversational_chain = require("./lib/conversational_chain");
var _error_handler = require("./utils/error_handler");
var _handle_stream_response = require("./utils/handle_stream_response");
var _types = require("./types");
var _get_chat_params = require("./lib/get_chat_params");
var _fetch_indices = require("./lib/fetch_indices");
var _is_not_nullish = require("../common/is_not_nullish");
var _models = require("../common/models");
var _errors = require("./lib/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createRetriever(esQuery) {
  return question => {
    try {
      const replacedQuery = esQuery.replace(/\"{query}\"/g, JSON.stringify(question));
      const query = JSON.parse(replacedQuery);
      return query;
    } catch (e) {
      throw Error("Failed to parse the Elasticsearch Query. Check Query to make sure it's valid.");
    }
  };
}
function defineRoutes({
  logger,
  router,
  getStartServices
}) {
  router.post({
    path: _types.APIRoutes.POST_QUERY_SOURCE_FIELDS,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      indices
    } = request.body;
    const fields = await (0, _fetch_query_source_fields.fetchFields)(client, indices);
    return response.ok({
      body: fields
    });
  }));
  router.post({
    path: _types.APIRoutes.POST_CHAT_MESSAGE,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        data: _configSchema.schema.object({
          connector_id: _configSchema.schema.string(),
          indices: _configSchema.schema.string(),
          prompt: _configSchema.schema.string(),
          citations: _configSchema.schema.boolean(),
          elasticsearch_query: _configSchema.schema.string(),
          summarization_model: _configSchema.schema.maybe(_configSchema.schema.string()),
          doc_size: _configSchema.schema.number(),
          source_fields: _configSchema.schema.string()
        }),
        messages: _configSchema.schema.any()
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const [{
      analytics
    }, {
      actions,
      cloud,
      inference
    }] = await getStartServices();
    const {
      client
    } = (await context.core).elasticsearch;
    const aiClient = (0, _assist.createAssist)({
      es_client: client.asCurrentUser
    });
    const {
      messages,
      data
    } = request.body;
    const {
      chatModel,
      chatPrompt,
      questionRewritePrompt,
      connector,
      summarizationModel
    } = await (0, _get_chat_params.getChatParams)({
      connectorId: data.connector_id,
      model: data.summarization_model,
      citations: data.citations,
      prompt: data.prompt
    }, {
      actions,
      inference,
      logger,
      request
    });
    let sourceFields = {};
    try {
      sourceFields = JSON.parse(data.source_fields);
      sourceFields = Object.keys(sourceFields).reduce((acc, key) => {
        // @ts-ignore
        acc[key] = sourceFields[key][0];
        return acc;
      }, {});
    } catch (e) {
      logger.error('Failed to parse the source fields', e);
      throw Error(e);
    }
    const model = _models.MODELS.find(m => m.model === summarizationModel);
    const modelPromptLimit = model === null || model === void 0 ? void 0 : model.promptTokenLimit;
    const chain = (0, _conversational_chain.ConversationalChain)({
      model: chatModel,
      rag: {
        index: data.indices,
        retriever: createRetriever(data.elasticsearch_query),
        content_field: sourceFields,
        size: Number(data.doc_size),
        inputTokensLimit: modelPromptLimit
      },
      prompt: chatPrompt,
      questionRewritePrompt
    });
    try {
      var _connector$config, _cloud$isCloudEnabled;
      const stream = await chain.stream(aiClient, messages);
      analytics.reportEvent(_events.sendMessageEvent.eventType, {
        connectorType: connector.actionTypeId + ((_connector$config = connector.config) !== null && _connector$config !== void 0 && _connector$config.apiProvider ? `-${connector.config.apiProvider}` : ''),
        model: summarizationModel !== null && summarizationModel !== void 0 ? summarizationModel : '',
        isCitationsEnabled: data.citations
      });
      return (0, _handle_stream_response.handleStreamResponse)({
        logger,
        stream,
        response,
        request,
        isCloud: (_cloud$isCloudEnabled = cloud === null || cloud === void 0 ? void 0 : cloud.isCloudEnabled) !== null && _cloud$isCloudEnabled !== void 0 ? _cloud$isCloudEnabled : false
      });
    } catch (e) {
      if (e instanceof _errors.ContextLimitError) {
        return response.badRequest({
          body: {
            message: _i18n.i18n.translate('xpack.searchPlayground.serverErrors.exceedsModelTokenLimit', {
              defaultMessage: 'Your request uses {approxPromptTokens} input tokens. This exceeds the model token limit of {modelLimit} tokens. Please try using a different model thats capable of accepting larger prompts or reducing the prompt by decreasing the size of the context documents. If you are unsure, please see our documentation.',
              values: {
                modelLimit: e.modelLimit,
                approxPromptTokens: e.currentTokens
              }
            })
          }
        });
      }
      logger.error('Failed to create the chat stream', e);
      if (typeof e === 'object') {
        return response.badRequest({
          body: {
            message: e.message
          }
        });
      }
      throw e;
    }
  }));

  // SECURITY: We don't apply any authorization tags to this route because all actions performed
  // on behalf of the user making the request and governed by the user's own cluster privileges.
  router.get({
    path: _types.APIRoutes.GET_INDICES,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      query: _configSchema.schema.object({
        search_query: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 0
        })
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      search_query: searchQuery,
      size
    } = request.query;
    const {
      client: {
        asCurrentUser
      }
    } = (await context.core).elasticsearch;
    const {
      indexNames
    } = await (0, _fetch_indices.fetchIndices)(asCurrentUser, searchQuery);
    const indexNameSlice = indexNames.slice(0, size).filter(_is_not_nullish.isNotNullish);
    return response.ok({
      body: {
        indices: indexNameSlice
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: _types.APIRoutes.POST_SEARCH_QUERY,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        search_query: _configSchema.schema.string(),
        elasticsearch_query: _configSchema.schema.string(),
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        size: _configSchema.schema.maybe(_configSchema.schema.number({
          defaultValue: 10,
          min: 0
        })),
        from: _configSchema.schema.maybe(_configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }))
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      elasticsearch_query: elasticsearchQuery,
      indices,
      size,
      from
    } = request.body;
    try {
      if (indices.length === 0) {
        return response.badRequest({
          body: {
            message: 'Indices cannot be empty'
          }
        });
      }
      const retriever = createRetriever(elasticsearchQuery)(request.body.search_query);
      const searchResult = await client.asCurrentUser.search({
        index: indices,
        retriever: retriever.retriever,
        from,
        size
      });
      const total = searchResult.hits.total ? typeof searchResult.hits.total === 'object' ? searchResult.hits.total.value : searchResult.hits.total : 0;
      return response.ok({
        body: {
          results: searchResult.hits.hits,
          pagination: {
            from,
            size,
            total
          }
        }
      });
    } catch (e) {
      logger.error('Failed to search the query', e);
      if (typeof e === 'object' && e.message) {
        return response.badRequest({
          body: {
            message: e.message
          }
        });
      }
      throw e;
    }
  }));
  router.post({
    path: _types.APIRoutes.GET_INDEX_MAPPINGS,
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_common.PLUGIN_ID]
      }
    },
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, (0, _error_handler.errorHandler)(logger)(async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      indices
    } = request.body;
    try {
      if (indices.length === 0) {
        return response.badRequest({
          body: {
            message: 'Indices cannot be empty'
          }
        });
      }
      const mappings = await client.asCurrentUser.indices.getMapping({
        index: indices
      });
      return response.ok({
        body: {
          mappings
        }
      });
    } catch (e) {
      logger.error('Failed to get index mappings', e);
      if (typeof e === 'object' && e.message) {
        return response.badRequest({
          body: {
            message: e.message
          }
        });
      }
      throw e;
    }
  }));
}