"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConversationalChain = ConversationalChain;
exports.contextLimitCheck = contextLimitCheck;
exports.getMessages = void 0;
exports.registerContextTokenCounts = registerContextTokenCounts;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _prompts = require("@langchain/core/prompts");
var _runnables = require("@langchain/core/runnables");
var _output_parsers = require("@langchain/core/output_parsers");
var _ai = require("ai");
var _langchain = require("@ai-sdk/langchain");
var _messages = require("@langchain/core/messages");
var _elasticsearch_retriever = require("./elasticsearch_retriever");
var _render_template = require("../utils/render_template");
var _get_citations = require("../utils/get_citations");
var _token_tracking = require("./token_tracking");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSerialisedMessages = chatHistory => {
  const formattedDialogueTurns = chatHistory.map(message => {
    if (message instanceof _messages.HumanMessage) {
      return `Human: ${message.content}`;
    } else if (message instanceof _messages.AIMessage) {
      return `Assistant: ${message.content}`;
    }
  });
  return formattedDialogueTurns.join('\n');
};
const getMessages = chatHistory => {
  return chatHistory.map(message => {
    if (message.role === 'human') {
      return new _messages.HumanMessage(message.content);
    } else if (message.role === 'assistant') {
      return new _messages.AIMessage(message.content);
    }
    return null;
  }).filter(message => message !== null);
};
exports.getMessages = getMessages;
const buildContext = docs => {
  const serializedDocs = docs.map((doc, i) => (0, _render_template.renderTemplate)(`
position: ${i + 1}
{pageContent}`, {
    pageContent: doc.pageContent,
    ...doc.metadata
  }));
  return serializedDocs.join('\n');
};
const messageContentToString = content => {
  if (typeof content === 'string') {
    return content;
  }
  if (Array.isArray(content)) {
    return content.map(part => {
      if (typeof part === 'string') {
        return part;
      }
      if (typeof part === 'object' && part !== null) {
        if ('text' in part && typeof part.text === 'string') {
          return part.text;
        }
      }
      return '';
    }).filter(Boolean).join('\n');
  }
  return '';
};
const isAIMessageChunk = chunk => {
  var _constructor;
  return Boolean(chunk && typeof chunk === 'object' && 'constructor' in chunk && ((_constructor = chunk.constructor) === null || _constructor === void 0 ? void 0 : _constructor.name) === 'AIMessageChunk');
};
function contextLimitCheck(modelLimit, prompt) {
  return async input => {
    if (!modelLimit) return input;
    const stringPrompt = await prompt.format(input);
    const approxPromptTokens = (0, _token_tracking.getTokenEstimate)(stringPrompt);
    const aboveContextLimit = approxPromptTokens > modelLimit;
    if (aboveContextLimit) {
      throw new _errors.ContextLimitError('Context exceeds the model limit', modelLimit, approxPromptTokens);
    }
    return input;
  };
}
const writeAnnotations = (writer, annotations) => {
  const data = Array.isArray(annotations) ? annotations : [annotations];
  writer.write({
    type: 'data-message_annotations',
    data
  });
};
function registerContextTokenCounts(writer) {
  return input => {
    writeAnnotations(writer, {
      type: 'context_token_count',
      count: (0, _token_tracking.getTokenEstimate)(input.context)
    });
    return input;
  };
}
class ConversationalChainFn {
  constructor(options) {
    (0, _defineProperty2.default)(this, "options", void 0);
    this.options = options;
  }
  async stream(client, msgs) {
    const uiStream = (0, _ai.createUIMessageStream)({
      execute: async ({
        writer
      }) => {
        var _lcMessages, _this$options, _this$options$rag;
        const messages = msgs !== null && msgs !== void 0 ? msgs : [];
        const lcMessages = getMessages(messages);
        const previousMessages = lcMessages.slice(0, -1);
        const lastMessageContent = (_lcMessages = lcMessages[lcMessages.length - 1]) === null || _lcMessages === void 0 ? void 0 : _lcMessages.content;
        const question = messageContentToString(lastMessageContent !== null && lastMessageContent !== void 0 ? lastMessageContent : '');
        const retrievedDocs = [];
        let hasWrittenCitations = false;
        let retrievalChain = _runnables.RunnableLambda.from(() => '');
        const chatHistory = getSerialisedMessages(previousMessages);
        if (this.options.rag) {
          var _this$options$rag$hit, _this$options$rag$siz;
          const retriever = new _elasticsearch_retriever.ElasticsearchRetriever({
            retriever: this.options.rag.retriever,
            index: this.options.rag.index,
            client: client.getClient(),
            content_field: this.options.rag.content_field,
            hit_doc_mapper: (_this$options$rag$hit = this.options.rag.hit_doc_mapper) !== null && _this$options$rag$hit !== void 0 ? _this$options$rag$hit : undefined,
            k: (_this$options$rag$siz = this.options.rag.size) !== null && _this$options$rag$siz !== void 0 ? _this$options$rag$siz : 3
          });
          retrievalChain = retriever.pipe(buildContext);
        }
        let standaloneQuestionChain = _runnables.RunnableLambda.from(input => {
          return input.question;
        });
        if (lcMessages.length > 1) {
          const questionRewritePromptTemplate = _prompts.PromptTemplate.fromTemplate(this.options.questionRewritePrompt);
          standaloneQuestionChain = _runnables.RunnableSequence.from([{
            context: input => input.chat_history,
            question: input => input.question
          }, questionRewritePromptTemplate, this.options.model, new _output_parsers.StringOutputParser()]).withConfig({
            metadata: {
              type: 'standalone_question'
            }
          });
        }
        const prompt = _prompts.ChatPromptTemplate.fromMessages([_prompts.SystemMessagePromptTemplate.fromTemplate(this.options.prompt), ...lcMessages]);
        const answerChain = _runnables.RunnableSequence.from([{
          context: _runnables.RunnableSequence.from([input => input.question, retrievalChain]),
          question: input => input.question
        }, _runnables.RunnableLambda.from(inputs => {
          writeAnnotations(writer, {
            type: 'search_query',
            question: inputs.question
          });
          return inputs;
        }), _runnables.RunnableLambda.from(contextLimitCheck((_this$options = this.options) === null || _this$options === void 0 ? void 0 : (_this$options$rag = _this$options.rag) === null || _this$options$rag === void 0 ? void 0 : _this$options$rag.inputTokensLimit, prompt)), _runnables.RunnableLambda.from(registerContextTokenCounts(writer)), prompt, this.options.model.withConfig({
          metadata: {
            type: 'question_answer_qa'
          }
        })]);
        const conversationalRetrievalQAChain = _runnables.RunnableSequence.from([{
          question: standaloneQuestionChain,
          chat_history: input => input.chat_history
        }, answerChain]);
        const lcStream = await conversationalRetrievalQAChain.stream({
          question,
          chat_history: chatHistory
        }, {
          callbacks: [{
            // callback for chat based models (OpenAI)
            handleChatModelStart(llm, msg, runId, parentRunId, extraParams, tags, metadata) {
              if ((metadata === null || metadata === void 0 ? void 0 : metadata.type) === 'question_answer_qa') {
                writeAnnotations(writer, [{
                  type: 'prompt_token_count',
                  count: (0, _token_tracking.getTokenEstimateFromMessages)(msg)
                }, {
                  type: 'search_query',
                  question
                }]);
              }
            },
            // callback for prompt based models (Bedrock uses ActionsClientLlm)
            handleLLMStart(llm, input, runId, parentRunId, extraParams, tags, metadata) {
              if ((metadata === null || metadata === void 0 ? void 0 : metadata.type) === 'question_answer_qa') {
                writeAnnotations(writer, {
                  type: 'prompt_token_count',
                  count: (0, _token_tracking.getTokenEstimate)(input[0])
                });
              }
            },
            handleRetrieverEnd(documents) {
              retrievedDocs.push(...documents);
              writeAnnotations(writer, {
                type: 'retrieved_docs',
                documents
              });
            },
            handleChainEnd(outputs) {
              if (hasWrittenCitations || !isAIMessageChunk(outputs)) {
                return;
              }
              hasWrittenCitations = true;
              writeAnnotations(writer, {
                type: 'citations',
                documents: (0, _get_citations.getCitations)(outputs.content, 'inline', retrievedDocs)
              });
            }
          }]
        });
        writer.merge((0, _langchain.toUIMessageStream)(lcStream));
      },
      onError: error => {
        if (error instanceof Error) {
          return error.message;
        }
        return 'An error occurred while processing the request';
      }
    });
    return uiStream.pipeThrough(new _ai.JsonToSseTransformStream());
  }
}
function ConversationalChain(options) {
  return new ConversationalChainFn(options);
}