"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Chat = void 0;
var _react = _interopRequireWildcard(require("react"));
var _reactHookForm = require("react-hook-form");
var _eui = require("@elastic/eui");
var _uuid = require("uuid");
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _constants = require("../analytics/constants");
var _use_auto_bottom_scroll = require("../hooks/use_auto_bottom_scroll");
var _chat_sidebar = require("./chat_sidebar");
var _use_chat = require("../hooks/use_chat");
var _types = require("../types");
var _message_list = require("./message_list/message_list");
var _question_input = require("./question_input");
var _telegram_icon = require("./telegram_icon");
var _transform_to_messages = require("../utils/transform_to_messages");
var _use_usage_tracker = require("../hooks/use_usage_tracker");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildFormData = formData => ({
  connector_id: formData[_types.ChatFormFields.summarizationModel].connectorId,
  prompt: formData[_types.ChatFormFields.prompt],
  indices: formData[_types.ChatFormFields.indices].join(),
  citations: formData[_types.ChatFormFields.citations],
  elasticsearch_query: JSON.stringify(formData[_types.ChatFormFields.elasticsearchQuery]),
  summarization_model: formData[_types.ChatFormFields.summarizationModel].value,
  source_fields: JSON.stringify(formData[_types.ChatFormFields.sourceFields]),
  doc_size: formData[_types.ChatFormFields.docSize]
});
const Chat = () => {
  const {
    euiTheme
  } = (0, _eui.useEuiTheme)();
  const {
    control,
    formState: {
      isValid,
      isSubmitting
    },
    resetField,
    handleSubmit,
    getValues
  } = (0, _reactHookForm.useFormContext)();
  const {
    messages,
    append,
    stop: stopRequest,
    setMessages,
    reload
  } = (0, _use_chat.useChat)();
  const messagesRef = (0, _use_auto_bottom_scroll.useAutoBottomScroll)();
  const [isRegenerating, setIsRegenerating] = (0, _react.useState)(false);
  const usageTracker = (0, _use_usage_tracker.useUsageTracker)();
  const onSubmit = async data => {
    await append({
      content: data.question,
      role: _types.MessageRole.user,
      createdAt: new Date()
    }, {
      data: buildFormData(data)
    });
    usageTracker === null || usageTracker === void 0 ? void 0 : usageTracker.click(_constants.AnalyticsEvents.chatQuestionSent);
    resetField(_types.ChatFormFields.question);
  };
  const handleStopRequest = () => {
    stopRequest();
    usageTracker === null || usageTracker === void 0 ? void 0 : usageTracker.click(_constants.AnalyticsEvents.chatRequestStopped);
  };
  const chatMessages = (0, _react.useMemo)(() => [{
    id: (0, _uuid.v4)(),
    role: _types.MessageRole.system,
    content: 'Welcome! Ask a question to get started.'
  }, ...(0, _transform_to_messages.transformFromChatMessages)(messages)], [messages]);
  const isToolBarActionsDisabled = (0, _react.useMemo)(() => chatMessages.length <= 1 || isRegenerating || isSubmitting, [chatMessages, isSubmitting, isRegenerating]);
  const regenerateMessages = async () => {
    setIsRegenerating(true);
    const formData = getValues();
    await reload({
      data: buildFormData(formData)
    });
    setIsRegenerating(false);
    usageTracker === null || usageTracker === void 0 ? void 0 : usageTracker.click(_constants.AnalyticsEvents.chatRegenerateMessages);
  };
  const handleClearChat = () => {
    setMessages([]);
    usageTracker === null || usageTracker === void 0 ? void 0 : usageTracker.click(_constants.AnalyticsEvents.chatCleared);
  };
  (0, _react.useEffect)(() => {
    usageTracker === null || usageTracker === void 0 ? void 0 : usageTracker.load(_constants.AnalyticsEvents.chatPageLoaded);
  }, [usageTracker]);
  return /*#__PURE__*/_react.default.createElement(_eui.EuiForm, {
    component: "form",
    css: {
      display: 'flex',
      flexGrow: 1
    },
    onSubmit: handleSubmit(onSubmit),
    "data-test-subj": "chatPage"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    gutterSize: "none"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: 2,
    css: {
      paddingTop: euiTheme.size.l,
      paddingBottom: euiTheme.size.l,
      // don't allow the chat to shrink below 66.6% of the screen
      flexBasis: 0,
      minWidth: '66.6%'
    }
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    className: "eui-fullHeight"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    className: "eui-yScroll",
    css: {
      paddingLeft: euiTheme.size.l,
      paddingRight: euiTheme.size.l
    },
    tabIndex: 0,
    ref: messagesRef
  }, /*#__PURE__*/_react.default.createElement(_message_list.MessageList, {
    messages: chatMessages
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false,
    css: {
      paddingLeft: euiTheme.size.l,
      paddingRight: euiTheme.size.l
    }
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiHorizontalRule, {
    margin: "none"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    responsive: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
    iconType: "sparkles",
    disabled: isToolBarActionsDisabled,
    onClick: regenerateMessages,
    size: "xs",
    "data-test-subj": "regenerateActionButton"
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.searchPlayground.chat.regenerateBtn",
    defaultMessage: "Regenerate"
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
    iconType: "refresh",
    disabled: isToolBarActionsDisabled,
    onClick: handleClearChat,
    size: "xs",
    "data-test-subj": "clearChatActionButton"
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.searchPlayground.chat.clearChatBtn",
    defaultMessage: "Clear chat"
  })))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_reactHookForm.Controller, {
    name: _types.ChatFormFields.question,
    control: control,
    defaultValue: "",
    rules: {
      required: true,
      validate: rule => !!(rule !== null && rule !== void 0 && rule.trim())
    },
    render: ({
      field
    }) => /*#__PURE__*/_react.default.createElement(_question_input.QuestionInput, {
      value: field.value,
      onChange: field.onChange,
      isDisabled: isSubmitting || isRegenerating,
      button: isSubmitting || isRegenerating ? /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
        "data-test-subj": "stopRequestButton",
        "aria-label": _i18n.i18n.translate('xpack.searchPlayground.chat.stopButtonAriaLabel', {
          defaultMessage: 'Stop request'
        }),
        display: "base",
        size: "s",
        iconType: "stop",
        onClick: handleStopRequest
      }) : /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
        "aria-label": _i18n.i18n.translate('xpack.searchPlayground.chat.sendButtonAriaLabel', {
          defaultMessage: 'Send a question'
        }),
        display: isValid ? 'base' : 'empty',
        size: "s",
        type: "submit",
        isLoading: isSubmitting,
        isDisabled: !isValid,
        iconType: _telegram_icon.TelegramIcon,
        "data-test-subj": "sendQuestionButton"
      })
    })
  })))), /*#__PURE__*/_react.default.createElement(_eui.EuiHideFor, {
    sizes: ['xs', 's']
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: 1,
    css: {
      flexBasis: 0,
      minWidth: '33.3%'
    }
  }, /*#__PURE__*/_react.default.createElement(_chat_sidebar.ChatSidebar, null)))));
};
exports.Chat = Chat;