"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JobCreator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _fieldTypes = require("@kbn/field-types");
var _mlQueryUtils = require("@kbn/ml-query-utils");
var _mlAnomalyUtils = require("@kbn/ml-anomaly-utils");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlParseInterval = require("@kbn/ml-parse-interval");
var _job_utils = require("../../../../../../common/util/job_utils");
var _index_utils = require("../../../../util/index_utils");
var _fields_utils = require("../../../../../../common/util/fields_utils");
var _default_configs = require("./util/default_configs");
var _job_runner = require("../job_runner");
var _new_job = require("../../../../../../common/constants/new_job");
var _general = require("./util/general");
var _filter_runtime_mappings = require("./util/filter_runtime_mappings");
var _calendar_service = require("../../../../services/calendar_service");
var _datafeed_utils = require("../../../../../../common/util/datafeed_utils");
var _object_utils = require("../../../../../../common/util/object_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class JobCreator {
  constructor(mlApi, newJobCapsService, indexPattern, savedSearch, query) {
    (0, _defineProperty2.default)(this, "_type", _new_job.JOB_TYPE.SINGLE_METRIC);
    (0, _defineProperty2.default)(this, "_indexPattern", void 0);
    (0, _defineProperty2.default)(this, "_savedSearch", void 0);
    (0, _defineProperty2.default)(this, "_indexPatternTitle", '');
    (0, _defineProperty2.default)(this, "_indexPatternDisplayName", '');
    (0, _defineProperty2.default)(this, "_job_config", void 0);
    (0, _defineProperty2.default)(this, "_calendars", void 0);
    (0, _defineProperty2.default)(this, "_datafeed_config", void 0);
    (0, _defineProperty2.default)(this, "_detectors", void 0);
    (0, _defineProperty2.default)(this, "_influencers", void 0);
    (0, _defineProperty2.default)(this, "_bucketSpanMs", 0);
    (0, _defineProperty2.default)(this, "_useDedicatedIndex", false);
    (0, _defineProperty2.default)(this, "_start", 0);
    (0, _defineProperty2.default)(this, "_end", 0);
    (0, _defineProperty2.default)(this, "_subscribers", []);
    (0, _defineProperty2.default)(this, "_aggs", []);
    (0, _defineProperty2.default)(this, "_fields", []);
    (0, _defineProperty2.default)(this, "_scriptFields", []);
    (0, _defineProperty2.default)(this, "_runtimeFields", []);
    (0, _defineProperty2.default)(this, "_runtimeMappings", null);
    (0, _defineProperty2.default)(this, "_filterRuntimeMappingsOnSave", true);
    (0, _defineProperty2.default)(this, "_aggregationFields", []);
    (0, _defineProperty2.default)(this, "_sparseData", false);
    (0, _defineProperty2.default)(this, "_stopAllRefreshPolls", {
      stop: false
    });
    (0, _defineProperty2.default)(this, "_wizardInitialized$", new _rxjs.BehaviorSubject(false));
    (0, _defineProperty2.default)(this, "wizardInitialized$", this._wizardInitialized$.asObservable());
    (0, _defineProperty2.default)(this, "mlApi", void 0);
    (0, _defineProperty2.default)(this, "newJobCapsService", void 0);
    this.mlApi = mlApi;
    this.newJobCapsService = newJobCapsService;
    this._indexPattern = indexPattern;
    this._savedSearch = savedSearch;
    const title = this._indexPattern.title;
    const name = this._indexPattern.getName();
    this._indexPatternDisplayName = name === title ? name : `${name} (${title})`;
    this._indexPatternTitle = title;
    this._job_config = (0, _default_configs.createEmptyJob)();
    this._calendars = [];
    this._datafeed_config = (0, _default_configs.createEmptyDatafeed)(this._indexPatternTitle);
    this._detectors = this._job_config.analysis_config.detectors;
    this._influencers = this._job_config.analysis_config.influencers;
    if (typeof indexPattern.timeFieldName === 'string') {
      this._job_config.data_description.time_field = indexPattern.timeFieldName;
    }
    this._extractRuntimeMappings();
    this._datafeed_config.query = query;
  }
  get type() {
    return this._type;
  }
  get savedSearch() {
    return this._savedSearch;
  }
  get dataView() {
    return this._indexPattern;
  }
  get dataViewId() {
    return this._indexPattern.id;
  }
  get indexPatternTitle() {
    return this._indexPatternTitle;
  }
  get indexPatternDisplayName() {
    return this._indexPatternDisplayName;
  }
  _addDetector(detector, agg, field) {
    this._detectors.push(detector);
    this._aggs.push(agg);
    this._fields.push(field);
    this._updateSparseDataDetectors();
  }
  _editDetector(detector, agg, field, index) {
    if (this._detectors[index] !== undefined) {
      this._detectors[index] = detector;
      this._aggs[index] = agg;
      this._fields[index] = field;
      this._updateSparseDataDetectors();
    }
  }
  _removeDetector(index) {
    this._detectors.splice(index, 1);
    this._aggs.splice(index, 1);
    this._fields.splice(index, 1);
  }
  removeAllDetectors() {
    this._detectors.length = 0;
    this._aggs.length = 0;
    this._fields.length = 0;
  }
  get savedSearchQuery() {
    return this._savedSearch ? (0, _index_utils.getQueryFromSavedSearchObject)(this._savedSearch) : null;
  }
  get detectors() {
    return this._detectors;
  }
  get aggregationsInDetectors() {
    return this._aggs;
  }
  getAggregation(index) {
    const agg = this._aggs[index];
    return agg !== undefined ? agg : null;
  }
  get aggregations() {
    return this._aggs;
  }
  getField(index) {
    const field = this._fields[index];
    return field !== undefined ? field : null;
  }
  get fields() {
    return this._fields;
  }
  set bucketSpan(bucketSpan) {
    this._job_config.analysis_config.bucket_span = bucketSpan;
    this._setBucketSpanMs(bucketSpan);
  }
  get bucketSpan() {
    return this._job_config.analysis_config.bucket_span;
  }
  _setBucketSpanMs(bucketSpan) {
    const bs = (0, _mlParseInterval.parseInterval)(bucketSpan, true);
    this._bucketSpanMs = bs === null ? 0 : bs.asMilliseconds();
  }
  get bucketSpanMs() {
    return this._bucketSpanMs;
  }
  addInfluencer(influencer) {
    if (this._influencers.includes(influencer) === false) {
      this._influencers.push(influencer);
    }
  }
  removeInfluencer(influencer) {
    const idx = this._influencers.indexOf(influencer);
    if (idx !== -1) {
      this._influencers.splice(idx, 1);
    }
  }
  removeAllInfluencers() {
    this._influencers.length = 0;
  }
  get influencers() {
    return this._influencers;
  }
  set jobId(jobId) {
    this._job_config.job_id = jobId;
    this._datafeed_config.job_id = jobId;
    this._datafeed_config.datafeed_id = (0, _job_utils.createDatafeedId)(jobId);
    if (this._useDedicatedIndex) {
      this._job_config.results_index_name = jobId;
    }
  }
  get jobId() {
    return this._job_config.job_id;
  }
  get datafeedId() {
    return this._datafeed_config.datafeed_id;
  }
  set description(description) {
    this._job_config.description = description;
  }
  get description() {
    var _this$_job_config$des;
    return (_this$_job_config$des = this._job_config.description) !== null && _this$_job_config$des !== void 0 ? _this$_job_config$des : '';
  }
  get groups() {
    var _this$_job_config$gro;
    return (_this$_job_config$gro = this._job_config.groups) !== null && _this$_job_config$gro !== void 0 ? _this$_job_config$gro : [];
  }
  set groups(groups) {
    this._job_config.groups = groups;
  }
  get calendars() {
    return this._calendars;
  }
  set calendars(calendars) {
    this._calendars = calendars;
  }
  _initModelPlotConfig() {
    // initialize configs to false if they are missing
    if (this._job_config.model_plot_config === undefined) {
      this._job_config.model_plot_config = {
        enabled: false
      };
    }
    if (this._job_config.model_plot_config !== undefined && this._job_config.model_plot_config.enabled === undefined) {
      this._job_config.model_plot_config.enabled = false;
    }
    if (this._job_config.model_plot_config !== undefined && this._job_config.model_plot_config.annotations_enabled === undefined) {
      this._job_config.model_plot_config.annotations_enabled = false;
    }
  }
  set modelPlot(enable) {
    this._initModelPlotConfig();
    this._job_config.model_plot_config.enabled = enable;
  }
  get modelPlot() {
    return this._job_config.model_plot_config !== undefined && this._job_config.model_plot_config.enabled === true;
  }
  set modelChangeAnnotations(enable) {
    this._initModelPlotConfig();
    this._job_config.model_plot_config.annotations_enabled = enable;
  }
  get modelChangeAnnotations() {
    var _this$_job_config$mod;
    return ((_this$_job_config$mod = this._job_config.model_plot_config) === null || _this$_job_config$mod === void 0 ? void 0 : _this$_job_config$mod.annotations_enabled) === true;
  }
  set useDedicatedIndex(enable) {
    this._useDedicatedIndex = enable;
    if (enable) {
      if (this._job_config.results_index_name === undefined) {
        // only set the results_index_name if it hasn't been set before.
        // this allows it to be overwritten in the advanced JSON editor.
        this._job_config.results_index_name = this._job_config.job_id;
      }
    } else {
      // @ts-expect-error The operand of a 'delete' operator must be optional
      delete this._job_config.results_index_name;
    }
  }
  get useDedicatedIndex() {
    return this._useDedicatedIndex;
  }
  set modelMemoryLimit(mml) {
    if (mml !== null) {
      if (this._job_config.analysis_limits === undefined) {
        this._job_config.analysis_limits = {};
      }
      this._job_config.analysis_limits.model_memory_limit = mml;
    } else {
      if (this._job_config.analysis_limits !== undefined) {
        delete this._job_config.analysis_limits.model_memory_limit;
        if ((0, _mlIsPopulatedObject.isPopulatedObject)(this._job_config.analysis_limits) === false) {
          delete this._job_config.analysis_limits;
        }
      }
    }
  }
  get modelMemoryLimit() {
    if (this._job_config.analysis_limits && this._job_config.analysis_limits.model_memory_limit !== undefined) {
      return `${this._job_config.analysis_limits.model_memory_limit}`;
    } else {
      return null;
    }
  }
  set summaryCountFieldName(fieldName) {
    if (fieldName !== null) {
      this._job_config.analysis_config.summary_count_field_name = fieldName;
    } else {
      delete this._job_config.analysis_config.summary_count_field_name;
    }
  }
  get summaryCountFieldName() {
    return this._job_config.analysis_config.summary_count_field_name || null;
  }
  set categorizationFieldName(fieldName) {
    if (fieldName !== null) {
      this._job_config.analysis_config.categorization_field_name = fieldName;
    } else {
      delete this._job_config.analysis_config.categorization_field_name;
    }
  }
  get categorizationFieldName() {
    return this._job_config.analysis_config.categorization_field_name || null;
  }
  addCategorizationFilter(filter) {
    if (this._job_config.analysis_config.categorization_filters === undefined) {
      this._job_config.analysis_config.categorization_filters = [];
    }
    const filters = this._job_config.analysis_config.categorization_filters;
    if (filters.includes(filter) === false) {
      filters.push(filter);
    }
  }
  removeCategorizationFilter(filter) {
    const filters = this._job_config.analysis_config.categorization_filters;
    if (filters !== undefined) {
      const idx = filters.indexOf(filter);
      if (idx !== -1) {
        filters.splice(idx, 1);
      }
      if (filters.length === 0) {
        this.removeCategorizationFilters();
      }
    }
  }
  removeCategorizationFilters() {
    delete this._job_config.analysis_config.categorization_filters;
  }
  get categorizationFilters() {
    return this._job_config.analysis_config.categorization_filters || null;
  }
  get timeFieldName() {
    return this._job_config.data_description.time_field;
  }
  set timeFieldName(fieldName) {
    this._job_config.data_description.time_field = fieldName;
  }
  get sparseData() {
    return this._sparseData;
  }
  set sparseData(sparseData) {
    this._sparseData = sparseData;
    this._updateSparseDataDetectors();
  }
  _updateSparseDataDetectors() {
    // loop through each detector, if the aggregation in the corresponding detector index is a count or sum
    // change the detector to be a non-zer or non-null count or sum.
    // note, the aggregations will always be a standard count or sum and not a non-null or non-zero version
    this._detectors.forEach((d, i) => {
      if (this._aggs[i] === undefined) {
        return;
      }
      switch (this._aggs[i].id) {
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.COUNT:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.NON_ZERO_COUNT : _mlAnomalyUtils.ML_JOB_AGGREGATION.COUNT;
          break;
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_COUNT:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_NON_ZERO_COUNT : _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_COUNT;
          break;
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_COUNT:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_NON_ZERO_COUNT : _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_COUNT;
          break;
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.SUM:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.NON_NULL_SUM : _mlAnomalyUtils.ML_JOB_AGGREGATION.SUM;
          break;
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_SUM:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_NON_NULL_SUM : _mlAnomalyUtils.ML_JOB_AGGREGATION.HIGH_SUM;
          break;
        case _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_SUM:
          d.function = this._sparseData ? _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_NON_NULL_SUM : _mlAnomalyUtils.ML_JOB_AGGREGATION.LOW_SUM;
          break;
      }
    });
  }

  /**
   * Extends assigned calendars with created job id.
   * @internal
   */
  async _updateCalendars() {
    if (this._calendars.length === 0) {
      return;
    }
    for (const calendar of this._calendars) {
      await _calendar_service.mlCalendarService.assignNewJobId(this.mlApi, calendar, this.jobId);
    }
  }
  setTimeRange(start, end) {
    this._start = start;
    this._end = end;
  }
  get start() {
    return this._start;
  }
  get end() {
    return this._end;
  }
  get query() {
    return this._datafeed_config.query;
  }
  set query(query) {
    this._datafeed_config.query = query;
  }
  get queryDelay() {
    // @ts-expect-error `estypes.Duration = string | -1 | 0;`
    return this._datafeed_config.query_delay || null;
  }
  set queryDelay(queryDelay) {
    if (queryDelay !== null) {
      this._datafeed_config.query_delay = queryDelay;
    } else {
      delete this._datafeed_config.query_delay;
    }
  }
  get frequency() {
    // @ts-expect-error `estypes.Duration = string | -1 | 0;`
    return this._datafeed_config.frequency || null;
  }
  set frequency(frequency) {
    if (frequency !== null) {
      this._datafeed_config.frequency = frequency;
    } else {
      delete this._datafeed_config.frequency;
    }
  }
  get scrollSize() {
    return this._datafeed_config.scroll_size || null;
  }
  set scrollSize(scrollSize) {
    if (scrollSize !== null) {
      this._datafeed_config.scroll_size = scrollSize;
    } else {
      delete this._datafeed_config.scroll_size;
    }
  }
  get indices() {
    return this._datafeed_config.indices;
  }
  set indices(indics) {
    this._datafeed_config.indices = indics;
  }
  get ignoreUnavailable() {
    var _this$_datafeed_confi;
    return !!((_this$_datafeed_confi = this._datafeed_config.indices_options) !== null && _this$_datafeed_confi !== void 0 && _this$_datafeed_confi.ignore_unavailable);
  }
  set ignoreUnavailable(ignore) {
    if (ignore === true) {
      if (this._datafeed_config.indices_options === undefined) {
        this._datafeed_config.indices_options = {};
      }
      this._datafeed_config.indices_options.ignore_unavailable = true;
    } else {
      if (this._datafeed_config.indices_options !== undefined) {
        delete this._datafeed_config.indices_options.ignore_unavailable;

        // if no other properties are set, remove indices_options
        if (Object.keys(this._datafeed_config.indices_options).length === 0) {
          delete this._datafeed_config.indices_options;
        }
      }
    }
  }
  get scriptFields() {
    return this._scriptFields;
  }
  get runtimeMappings() {
    return this._runtimeMappings;
  }
  get runtimeFields() {
    return this._runtimeFields;
  }
  get aggregationFields() {
    return this._aggregationFields;
  }
  get additionalFields() {
    return [...this._scriptFields, ...this._runtimeFields, ...this._aggregationFields];
  }
  get subscribers() {
    return this._subscribers;
  }
  async createAndStartJob() {
    try {
      await this.createJob();
      await this.createDatafeed();
      const jobRunner = await this.startDatafeed();
      return jobRunner;
    } catch (error) {
      throw error;
    }
  }
  async createJob() {
    try {
      await this.mlApi.addJob({
        jobId: this._job_config.job_id,
        job: this._job_config
      });
      await this._updateCalendars();
    } catch (error) {
      throw error;
    }
  }
  async createDatafeed() {
    try {
      const tempDatafeed = this._getDatafeedWithFilteredRuntimeMappings();
      const jobId = this._job_config.job_id;
      const datafeedId = (0, _job_utils.createDatafeedId)(jobId);
      tempDatafeed.job_id = jobId;
      return this.mlApi.addDatafeed({
        datafeedId,
        datafeedConfig: tempDatafeed
      });
    } catch (error) {
      throw error;
    }
  }

  // create a jobRunner instance, start it and return it
  async startDatafeed() {
    const jobRunner = new _job_runner.JobRunner(this);
    await jobRunner.startDatafeed();
    return jobRunner;
  }
  _getDatafeedWithFilteredRuntimeMappings() {
    if (this._filterRuntimeMappingsOnSave === false) {
      return this._datafeed_config;
    }
    const {
      runtime_mappings: filteredRuntimeMappings
    } = (0, _filter_runtime_mappings.filterRuntimeMappings)(this._job_config, this._datafeed_config);
    return {
      ...this._datafeed_config,
      runtime_mappings: Object.keys(filteredRuntimeMappings).length > 0 ? filteredRuntimeMappings : undefined
    };
  }
  subscribeToProgress(func) {
    this._subscribers.push(func);
  }
  get jobConfig() {
    return this._job_config;
  }
  get datafeedConfig() {
    return this._datafeed_config;
  }
  get stopAllRefreshPolls() {
    return this._stopAllRefreshPolls;
  }
  forceStopRefreshPolls() {
    this._stopAllRefreshPolls.stop = true;
  }
  _setCustomSetting(setting, value) {
    if (value === null) {
      // if null is passed in, delete the custom setting
      if (this._job_config.custom_settings !== undefined && this._job_config.custom_settings[setting] !== undefined) {
        delete this._job_config.custom_settings[setting];
        if (Object.keys(this._job_config.custom_settings).length === 0) {
          // clean up custom_settings if there's nothing else in there
          delete this._job_config.custom_settings;
        }
      }
    } else {
      if (this._job_config.custom_settings === undefined) {
        // if custom_settings doesn't exist, create it.
        this._job_config.custom_settings = {
          [setting]: value
        };
      } else {
        // @ts-ignore
        this._job_config.custom_settings[setting] = value;
      }
    }
  }
  _getCustomSetting(setting) {
    if (this._job_config.custom_settings !== undefined && this._job_config.custom_settings[setting] !== undefined) {
      return this._job_config.custom_settings[setting];
    }
    return null;
  }
  set createdBy(createdBy) {
    this._setCustomSetting('created_by', createdBy);
  }
  get createdBy() {
    return this._getCustomSetting('created_by');
  }
  set customUrls(customUrls) {
    this._setCustomSetting('custom_urls', customUrls);
  }
  get customUrls() {
    return this._getCustomSetting('custom_urls');
  }
  get formattedJobJson() {
    return JSON.stringify(this._job_config, null, 2);
  }
  get formattedDatafeedJson() {
    return JSON.stringify(this._datafeed_config, null, 2);
  }
  set filterRuntimeMappingsOnSave(filter) {
    this._filterRuntimeMappingsOnSave = filter;
  }
  get filterRuntimeMappingsOnSave() {
    return this._filterRuntimeMappingsOnSave;
  }
  _initPerPartitionCategorization() {
    var _this$_job_config$ana, _this$_job_config$ana2;
    if (this._job_config.analysis_config.per_partition_categorization === undefined) {
      this._job_config.analysis_config.per_partition_categorization = {};
    }
    if (((_this$_job_config$ana = this._job_config.analysis_config.per_partition_categorization) === null || _this$_job_config$ana === void 0 ? void 0 : _this$_job_config$ana.enabled) === undefined) {
      this._job_config.analysis_config.per_partition_categorization.enabled = false;
    }
    if (((_this$_job_config$ana2 = this._job_config.analysis_config.per_partition_categorization) === null || _this$_job_config$ana2 === void 0 ? void 0 : _this$_job_config$ana2.stop_on_warn) === undefined) {
      this._job_config.analysis_config.per_partition_categorization.stop_on_warn = false;
    }
  }
  get perPartitionCategorization() {
    var _this$_job_config$ana3;
    return ((_this$_job_config$ana3 = this._job_config.analysis_config.per_partition_categorization) === null || _this$_job_config$ana3 === void 0 ? void 0 : _this$_job_config$ana3.enabled) === true;
  }
  set perPartitionCategorization(enabled) {
    this._initPerPartitionCategorization();
    this._job_config.analysis_config.per_partition_categorization.enabled = enabled;
  }
  get perPartitionStopOnWarn() {
    var _this$_job_config$ana4;
    return ((_this$_job_config$ana4 = this._job_config.analysis_config.per_partition_categorization) === null || _this$_job_config$ana4 === void 0 ? void 0 : _this$_job_config$ana4.stop_on_warn) === true;
  }
  set perPartitionStopOnWarn(enabled) {
    this._initPerPartitionCategorization();
    this._job_config.analysis_config.per_partition_categorization.stop_on_warn = enabled;
  }
  _extractRuntimeMappings() {
    const runtimeFieldMap = this._indexPattern.toSpec().runtimeFieldMap;
    if (runtimeFieldMap !== undefined) {
      if (this._datafeed_config.runtime_mappings === undefined) {
        this._datafeed_config.runtime_mappings = {};
      }
      Object.entries(runtimeFieldMap).forEach(([key, val]) => {
        this._datafeed_config.runtime_mappings[key] = val;
      });
    }
    this._populateRuntimeFields();
  }
  _populateRuntimeFields() {
    var _this$_datafeed_confi2;
    this._runtimeFields = [];
    this._runtimeMappings = (_this$_datafeed_confi2 = this._datafeed_config.runtime_mappings) !== null && _this$_datafeed_confi2 !== void 0 ? _this$_datafeed_confi2 : null;
    if (this._runtimeMappings !== null) {
      const tempRuntimeFields = Object.entries(this._runtimeMappings).map(([id, runtimeField]) => ({
        id,
        name: id,
        type: Array.isArray(runtimeField) ? runtimeField[0].type : runtimeField.type,
        aggregatable: true,
        counter: false,
        aggs: [],
        runtimeField
      }));
      const aggs = (0, _lodash.cloneDeep)([..._mlAnomalyUtils.mlJobAggregations, ..._mlAnomalyUtils.mlJobAggregationsWithoutEsEquivalent]);
      this._runtimeFields = (0, _fields_utils.combineFieldsAndAggs)(tempRuntimeFields, aggs, {}).fields;
    }
  }
  _populateScriptFields() {
    this._scriptFields = [];
    if (this._datafeed_config.script_fields !== undefined) {
      this._scriptFields = Object.keys(this._datafeed_config.script_fields).map(f => ({
        id: f,
        name: f,
        type: _fieldTypes.ES_FIELD_TYPES.KEYWORD,
        aggregatable: true,
        counter: false
      }));
    }
  }

  // load the start and end times for the selected index
  // and apply them to the job creator
  async autoSetTimeRange(excludeFrozenData = true) {
    const {
      start,
      end
    } = await this.mlApi.getTimeFieldRange({
      index: this._indexPatternTitle,
      timeFieldName: this.timeFieldName,
      query: excludeFrozenData ? (0, _mlQueryUtils.addExcludeFrozenToQuery)(this.query) : this.query,
      runtimeMappings: this.datafeedConfig.runtime_mappings,
      indicesOptions: this.datafeedConfig.indices_options
    });
    this.setTimeRange(start, end);
  }
  _overrideConfigs(job, datafeed) {
    this._job_config = job;
    this._datafeed_config = datafeed;
    this._detectors = this._job_config.analysis_config.detectors;
    this._influencers = this._job_config.analysis_config.influencers;
    if (this._job_config.groups === undefined) {
      this._job_config.groups = [];
    }
    if (this._job_config.analysis_config.influencers !== undefined) {
      this._job_config.analysis_config.influencers.forEach(i => this.addInfluencer(i));
    }
    if (this._job_config.results_index_name !== undefined && this._job_config.results_index_name !== _new_job.SHARED_RESULTS_INDEX_NAME) {
      this.useDedicatedIndex = true;
    }
    this._populateScriptFields();
    this._populateRuntimeFields();
    this._aggregationFields = [];
    const aggs = (0, _datafeed_utils.getDatafeedAggregations)(this._datafeed_config);
    if (aggs !== undefined) {
      const aggBucketsName = (0, _object_utils.getFirstKeyInObject)(aggs);
      if (aggBucketsName !== undefined && aggs[aggBucketsName] !== undefined) {
        const buckets = aggs[aggBucketsName];
        (0, _general.collectAggs)(buckets, this._aggregationFields);
      }
    }
  }
}
exports.JobCreator = JobCreator;