"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCelInputRoutes = registerCelInputRoutes;
var _server = require("@kbn/data-plugin/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("../../common");
var _constants = require("../constants");
var _cel = require("../graphs/cel");
var _llm = require("../util/llm");
var _route_validation = require("../util/route_validation");
var _with_availability = require("./with_availability");
var _errors = require("../lib/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerCelInputRoutes(router) {
  router.versioned.post({
    path: _common.CEL_INPUT_GRAPH_PATH,
    access: 'internal',
    options: {
      timeout: {
        idleSocket: _constants.ROUTE_HANDLER_TIMEOUT
      }
    }
  }).addVersion({
    version: '1',
    security: {
      authz: {
        requiredPrivileges: [_constants.FLEET_ALL_ROLE, _constants.INTEGRATIONS_ALL_ROLE]
      }
    },
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_common.CelInputRequestBody)
      }
    }
  }, (0, _with_availability.withAvailability)(async (context, req, res) => {
    const {
      dataStreamName,
      apiDefinition,
      langSmithOptions
    } = req.body;
    const {
      getStartServices,
      logger
    } = await context.integrationAssistant;
    const [, {
      actions: actionsPlugin
    }] = await getStartServices();
    try {
      var _connector$config, _langSmithOptions$pro;
      const actionsClient = await actionsPlugin.getActionsClientWithRequest(req);
      const connector = await actionsClient.get({
        id: req.body.connectorId
      });
      const abortSignal = (0, _server.getRequestAbortedSignal)(req.events.aborted$);
      const actionTypeId = connector.actionTypeId;
      const llmType = (0, _llm.getLLMType)(actionTypeId);
      const llmClass = (0, _llm.getLLMClass)(llmType);
      const model = new llmClass({
        actionsClient,
        connectorId: connector.id,
        logger,
        llmType,
        model: (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.defaultModel,
        temperature: 0.05,
        maxTokens: 4096,
        signal: abortSignal,
        streaming: false,
        telemetryMetadata: {
          pluginId: 'automatic_import'
        }
      });
      const parameters = {
        dataStreamName,
        apiDefinition
      };
      const options = {
        callbacks: [new _apm.APMTracer({
          projectName: (_langSmithOptions$pro = langSmithOptions === null || langSmithOptions === void 0 ? void 0 : langSmithOptions.projectName) !== null && _langSmithOptions$pro !== void 0 ? _langSmithOptions$pro : 'default'
        }, logger), ...(0, _langsmith.getLangSmithTracer)({
          ...langSmithOptions,
          logger
        })]
      };
      const graph = await (0, _cel.getCelGraph)({
        model
      });
      const results = await graph.withConfig({
        runName: 'CEL'
      }).invoke(parameters, options);
      return res.ok({
        body: _common.CelInputResponse.parse(results)
      });
    } catch (e) {
      if ((0, _errors.isErrorThatHandlesItsOwnResponse)(e)) {
        return e.sendResponse(res);
      }
      return res.badRequest({
        body: e
      });
    }
  }));
}