"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.columnsFromHeader = columnsFromHeader;
exports.generateColumnNames = generateColumnNames;
exports.prefixColumns = prefixColumns;
exports.toSafeColumnName = toSafeColumnName;
exports.totalColumnCount = totalColumnCount;
exports.upperBoundForColumnCount = upperBoundForColumnCount;
exports.yieldUniqueColumnNames = yieldUniqueColumnNames;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Estimates from above the number of columns in the CSV samples.
function upperBoundForColumnCount(csvSamples) {
  return Math.max(0, ...csvSamples.map(sample => sample.split(',').length));
}

// Generates a list of temporary column names.
function generateColumnNames(count) {
  return Array.from({
    length: count
  }).map((_, i) => `column${i + 1}`);
}

// Converts a column name into a safe one to use in the `if ctx...` clause.
// Result must pass rules at https://www.elastic.co/guide/en/elasticsearch/painless/8.15/painless-identifiers.html
function toSafeColumnName(columnName) {
  if (typeof columnName === 'number') {
    return `Column${columnName}`;
  }
  if (typeof columnName !== 'string') {
    return undefined;
  }
  if (columnName.length === 0) {
    return undefined;
  }
  const safeName = columnName.replace(/[^a-zA-Z0-9_]/g, '_');
  return /^[0-9]/.test(safeName) ? `Column${safeName}` : safeName;
}
// Returns the column list from a header row. We skip values that are not strings.

function columnsFromHeader(tempColumnNames, headerObject) {
  const maxIndex = tempColumnNames.findLastIndex(columnName => headerObject[columnName] !== undefined);
  return tempColumnNames.slice(0, maxIndex + 1).map(columnName => headerObject[columnName]).map(toSafeColumnName);
}
// Count the number of columns actually present in the rows.

function totalColumnCount(tempColumnNames, csvRows) {
  return Math.max(-1, ...csvRows.map(row => tempColumnNames.findLastIndex(columnName => row[columnName] !== undefined))) + 1;
}
// Prefixes each column with the provided prefixes, separated by a period.
function prefixColumns(columns, prefixes) {
  return columns.map(column => [...prefixes, column].join('.'));
}
/**
 * Generates a list of unique column names based on preferred and fallback names.
 *
 * The preferred names are used first, followed by the fallback names. It is required that
 * there are enough fallback names to cover the number of unique column names needed.
 *
 * The resulting column names are guaranteed to be unique. If a column name is already in use,
 * a postfix like _2, _3 and so on is added to the name to make it unique.
 *
 * @generator
 * @param {number} count - The number of unique column names to generate.
 * @param {Array<Array<string | undefined>>} preferredNames - A 2D array where each sub-array contains a list of names.
 * @param {string[]} fallbackNames - An array of fallback names to use if no preferred name is defined.
 * @yields {string} - A sequence of column names, such that no two are the same.
 */

function* yieldUniqueColumnNames(count, preferredNames, fallbackNames) {
  const knownNames = new Set();
  for (let i = 0; i < count; i++) {
    let selectedName = fallbackNames[i];
    for (const nameList of preferredNames) {
      const name = nameList[i];
      if (name) {
        selectedName = name;
        break;
      }
    }
    let postfixString = '';
    if (knownNames.has(selectedName)) {
      for (let postfix = 2;; postfix++) {
        postfixString = `_${postfix}`;
        if (!knownNames.has(selectedName + postfixString)) {
          break;
        }
      }
    }
    selectedName += postfixString;
    knownNames.add(selectedName);
    yield selectedName;
  }
}