"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bedrockClaudeAdapter = void 0;
var _rxjs = require("rxjs");
var _inferenceCommon = require("@kbn/inference-common");
var _utilUtf = require("@smithy/util-utf8");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _mlIsDefined = require("@kbn/ml-is-defined");
var _utils = require("../../utils");
var _serde_eventstream_into_observable = require("./serde_eventstream_into_observable");
var _process_completion_chunks = require("./process_completion_chunks");
var _prompts = require("./prompts");
var _convert_tools = require("./convert_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bedrockClaudeAdapter = exports.bedrockClaudeAdapter = {
  chatComplete: ({
    executor,
    system = 'You are a helpful assistant for Elastic.',
    messages,
    toolChoice,
    tools,
    temperature = 0,
    modelName,
    abortSignal,
    metadata
  }) => {
    const noToolUsage = toolChoice === _inferenceCommon.ToolChoiceType.none;
    const converseMessages = messagesToBedrock(messages).map(message => ({
      role: message.role,
      content: message.rawContent
    }));
    const systemMessage = noToolUsage ? [{
      text: (0, _prompts.addNoToolUsageDirective)(system)
    }] : [{
      text: system
    }];
    const bedRockTools = noToolUsage ? [] : (0, _convert_tools.toolsToConverseBedrock)(tools, messages);
    const subActionParams = {
      system: systemMessage,
      messages: converseMessages,
      tools: bedRockTools !== null && bedRockTools !== void 0 && bedRockTools.length ? bedRockTools : undefined,
      toolChoice: (0, _convert_tools.toolChoiceToConverse)(toolChoice),
      temperature,
      model: modelName,
      stopSequences: ['\n\nHuman:'],
      signal: abortSignal
    };
    return (0, _rxjs.defer)(async () => {
      const res = await executor.invoke({
        subAction: 'converseStream',
        subActionParams
      });
      const result = res.data;
      return {
        ...res,
        data: result === null || result === void 0 ? void 0 : result.stream
      };
    }).pipe((0, _utils.handleConnectorResponse)({
      processStream: _serde_eventstream_into_observable.serdeEventstreamIntoObservable
    }), (0, _rxjs.tap)(eventData => {
      if ((0, _mlIsPopulatedObject.isPopulatedObject)(eventData, ['modelStreamErrorException'])) {
        throw (0, _inferenceCommon.createInferenceInternalError)(eventData.modelStreamErrorException.originalMessage);
      }
    }), (0, _rxjs.filter)(value => {
      return typeof value === 'object' && !!value;
    }), (0, _rxjs.map)(message => {
      const key = Object.keys(message)[0];
      if (key && (0, _mlIsPopulatedObject.isPopulatedObject)(message, [key])) {
        return {
          type: key,
          body: JSON.parse((0, _utilUtf.toUtf8)(message[key].body))
        };
      }
    }), (0, _rxjs.filter)(value => !!value), (0, _process_completion_chunks.processConverseCompletionChunks)());
  }
};
const messagesToBedrock = messages => {
  const converseMessages = messages.map(message => {
    switch (message.role) {
      case _inferenceCommon.MessageRole.User:
        return {
          role: 'user',
          rawContent: (typeof message.content === 'string' ? [message.content] : message.content).map(contentPart => {
            var _contentPart$source;
            if (typeof contentPart === 'string') {
              return {
                text: contentPart,
                type: 'text'
              };
            } else if (contentPart.type === 'text') {
              return {
                text: contentPart.text,
                type: 'text'
              };
            }
            if ((_contentPart$source = contentPart.source) !== null && _contentPart$source !== void 0 && _contentPart$source.data) {
              const imageBlock = {
                // Convert mimetype = 'image/png' to 'png'
                // https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ImageBlock.html
                format: contentPart.source.mimeType.split('/')[1],
                source: {
                  bytes: new TextEncoder().encode(contentPart.source.data)
                }
              };
              return {
                image: imageBlock
              };
            }
          }).filter(_mlIsDefined.isDefined)
        };
      case _inferenceCommon.MessageRole.Assistant:
        return {
          role: 'assistant',
          rawContent: [...(message.content ? [{
            type: 'text',
            text: message.content
          }] : []), ...(message.toolCalls ? message.toolCalls.map(toolCall => {
            return {
              toolUse: {
                toolUseId: toolCall.toolCallId,
                name: toolCall.function.name,
                input: 'arguments' in toolCall.function ? toolCall.function.arguments : {}
              }
            };
          }) : [])]
        };
      case _inferenceCommon.MessageRole.Tool:
        return {
          role: 'user',
          rawContent: [{
            toolResult: {
              toolUseId: message.toolCallId,
              content: [typeof message.response === 'string' ? {
                text: message.response
              } : {
                json: message.response
              }]
            }
          }]
        };
    }
  });

  // Combine consecutive user tool result messages into a single message. This format is required by Bedrock.
  const combinedConverseMessages = converseMessages.reduce((acc, curr) => {
    var _lastMessage$rawConte, _curr$rawContent;
    const lastMessage = acc[acc.length - 1];
    if (lastMessage && lastMessage.role === 'user' && (_lastMessage$rawConte = lastMessage.rawContent) !== null && _lastMessage$rawConte !== void 0 && _lastMessage$rawConte.some(c => 'toolResult' in c) && curr.role === 'user' && (_curr$rawContent = curr.rawContent) !== null && _curr$rawContent !== void 0 && _curr$rawContent.some(c => 'toolResult' in c)) {
      lastMessage.rawContent = lastMessage.rawContent.concat(curr.rawContent);
    } else {
      acc.push(curr);
    }
    return acc;
  }, []);
  return combinedConverseMessages;
};