"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FleetArtifactsClient = void 0;
var _errors = require("../../errors");
var _mappings = require("./mappings");
var _artifacts = require("./artifacts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Exposes an interface for access artifacts from within the context of a single integration (`packageName`)
 */
class FleetArtifactsClient {
  constructor(esClient, packageName) {
    this.esClient = esClient;
    this.packageName = packageName;
    if (!packageName) {
      throw new _errors.ArtifactsClientError('packageName is required');
    }
  }
  validate(artifact) {
    if (artifact.packageName !== this.packageName) {
      throw new _errors.ArtifactsClientAccessDeniedError(artifact.packageName, this.packageName);
    }
    return artifact;
  }

  /**
   * Creates a `kuery` string using the provided value on input that is bound to the integration package
   * @param kuery
   * @internal
   */
  buildFilter(kuery) {
    return `(package_name: "${this.packageName}")${kuery ? ` AND ${kuery}` : ''}`;
  }
  async getArtifact(id) {
    const artifact = await (0, _artifacts.getArtifact)(this.esClient, id);
    return artifact ? this.validate(artifact) : undefined;
  }

  /**
   * Creates a new artifact. Content will be compress and stored in binary form.
   */
  async createArtifact({
    content,
    type = '',
    identifier = this.packageName
  }) {
    const encodedMetaData = await this.encodeContent(content);
    const newArtifactData = {
      type,
      identifier,
      packageName: this.packageName,
      encryptionAlgorithm: 'none',
      relative_url: (0, _mappings.relativeDownloadUrlFromArtifact)({
        identifier,
        decodedSha256: encodedMetaData.decodedSha256
      }),
      ...encodedMetaData
    };
    return (0, _artifacts.createArtifact)(this.esClient, newArtifactData);
  }
  async bulkCreateArtifacts(optionsList) {
    const newArtifactsData = [];
    for (const options of optionsList) {
      const {
        content,
        type = '',
        identifier = this.packageName
      } = options;
      const encodedMetaData = await this.encodeContent(content);
      const newArtifactData = {
        type,
        identifier,
        packageName: this.packageName,
        encryptionAlgorithm: 'none',
        relative_url: (0, _mappings.relativeDownloadUrlFromArtifact)({
          identifier,
          decodedSha256: encodedMetaData.decodedSha256
        }),
        ...encodedMetaData
      };
      newArtifactsData.push(newArtifactData);
    }
    return (0, _artifacts.bulkCreateArtifacts)(this.esClient, newArtifactsData);
  }
  async deleteArtifact(id) {
    // get the artifact first, which will also ensure its validated
    const artifact = await this.getArtifact(id);
    if (artifact) {
      await (0, _artifacts.deleteArtifact)(this.esClient, id);
    }
  }
  async bulkDeleteArtifacts(ids) {
    const idsMappedWithPackageName = ids.map(id => (0, _mappings.uniqueIdFromId)(id, this.packageName));
    return await (0, _artifacts.bulkDeleteArtifacts)(this.esClient, idsMappedWithPackageName);
  }

  /**
   * Get a list of artifacts. A few things to note:
   * - if wanting to get ALL artifacts, consider using instead the `fetchAll()` method instead
   *   as it will property return data past the 10k ES limitation
   * - when using the `kuery` filtering param, all filters property names should match the
   *   internal attribute names in the index
   */
  async listArtifacts({
    kuery,
    ...options
  } = {}) {
    return (0, _artifacts.listArtifacts)(this.esClient, {
      ...options,
      kuery: this.buildFilter(kuery)
    });
  }

  /**
   * Returns an `AsyncIterable` object that can be used to iterate over all artifacts
   *
   * @param options
   *
   * @example
   * async () => {
   *   for await (const artifacts of fleetArtifactsClient.fetchAll()) {
   *     // artifacts === first page of items
   *   }
   * }
   */
  fetchAll({
    kuery,
    ...options
  } = {}) {
    return (0, _artifacts.fetchAllArtifacts)(this.esClient, {
      ...options,
      kuery: this.buildFilter(kuery)
    });
  }
  generateHash(content) {
    return (0, _artifacts.generateArtifactContentHash)(content);
  }
  async encodeContent(content) {
    return (0, _artifacts.encodeArtifactContent)(content);
  }
}
exports.FleetArtifactsClient = FleetArtifactsClient;