"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFileHandler = void 0;
var _path = _interopRequireDefault(require("path"));
var _mimeTypes = _interopRequireDefault(require("mime-types"));
var _fs = require("@kbn/fs");
var _packages = require("../../services/epm/packages");
var _storage = require("../../services/epm/archive/storage");
var _bundled_packages = require("../../services/epm/packages/bundled_packages");
var _registry = require("../../services/epm/registry");
var _archive = require("../../services/epm/archive");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CACHE_CONTROL_10_MINUTES_HEADER = {
  'cache-control': 'max-age=600'
};
const ALLOWED_MIME_TYPES = ['image/svg+xml', 'image/jpeg', 'image/png', 'image/gif', 'application/json', 'application/yaml', 'text/plain', 'text/markdown', 'text/yaml'];
function validateContentTypeIsAllowed(contentType) {
  if (!ALLOWED_MIME_TYPES.includes(contentType.split(';')[0])) {
    throw new _errors.FleetError(`File content type "${contentType}" is not allowed to be retrieved`, 400);
  }
}
function sanitize(contentType, buffer) {
  if (contentType === 'image/svg+xml') {
    return (0, _fs.sanitizeSvg)(buffer);
  }
  return buffer;
}
const getFileHandler = async (context, request, response) => {
  const {
    pkgName,
    pkgVersion,
    filePath
  } = request.params;
  const savedObjectsClient = (await context.fleet).internalSoClient;
  const installation = await (0, _packages.getInstallation)({
    savedObjectsClient,
    pkgName
  });
  const isPackageInstalled = pkgVersion === (installation === null || installation === void 0 ? void 0 : installation.version);
  const assetPath = `${pkgName}-${pkgVersion}/${filePath}`;
  if (isPackageInstalled) {
    const storedAsset = await (0, _storage.getAsset)({
      savedObjectsClient,
      path: assetPath
    });
    if (!storedAsset) {
      return response.custom({
        body: `installed package file not found: ${filePath}`,
        statusCode: 404
      });
    }
    const contentType = storedAsset.media_type;
    validateContentTypeIsAllowed(contentType);
    let buffer = storedAsset.data_utf8 ? Buffer.from(storedAsset.data_utf8, 'utf8') : Buffer.from(storedAsset.data_base64, 'base64');
    if (!contentType) {
      return response.custom({
        body: `unknown content type for file: ${filePath}`,
        statusCode: 400
      });
    }

    // @ts-expect-error upgrade typescript v5.9.3
    buffer = sanitize(contentType, buffer);
    return response.custom({
      body: buffer,
      statusCode: 200,
      headers: {
        ...CACHE_CONTROL_10_MINUTES_HEADER,
        'content-type': contentType
      }
    });
  }
  const bundledPackage = await (0, _bundled_packages.getBundledPackageByPkgKey)((0, _registry.pkgToPkgKey)({
    name: pkgName,
    version: pkgVersion
  }));
  if (bundledPackage) {
    var _bufferEntries$find;
    const bufferEntries = await (0, _archive.unpackArchiveEntriesIntoMemory)(await bundledPackage.getBuffer(), 'application/zip');
    const fileBuffer = (_bufferEntries$find = bufferEntries.find(entry => entry.path === assetPath)) === null || _bufferEntries$find === void 0 ? void 0 : _bufferEntries$find.buffer;
    if (!fileBuffer) {
      return response.custom({
        body: `bundled package file not found: ${filePath}`,
        statusCode: 404
      });
    }

    // if storedAsset is not available, fileBuffer *must* be
    // b/c we error if we don't have at least one, and storedAsset is the least likely
    let buffer = fileBuffer;
    const contentType = _mimeTypes.default.contentType(_path.default.extname(assetPath));
    if (!contentType) {
      return response.custom({
        body: `unknown content type for file: ${filePath}`,
        statusCode: 400
      });
    }
    validateContentTypeIsAllowed(contentType);
    buffer = sanitize(contentType, buffer);
    return response.custom({
      body: buffer,
      statusCode: 200,
      headers: {
        ...CACHE_CONTROL_10_MINUTES_HEADER,
        'content-type': contentType
      }
    });
  } else {
    const registryResponse = await (0, _packages.getFile)(pkgName, pkgVersion, filePath);
    const headersToProxy = ['content-type'];
    const proxiedHeaders = headersToProxy.reduce((headers, knownHeader) => {
      const value = registryResponse.headers.get(knownHeader);
      if (value !== null) {
        headers[knownHeader] = value;
      }
      return headers;
    }, {});
    if (!proxiedHeaders['content-type'] || typeof proxiedHeaders['content-type'] !== 'string') {
      throw new _errors.FleetError(`unknown content type for file: ${filePath}`);
    }
    validateContentTypeIsAllowed(proxiedHeaders['content-type']);
    return response.custom({
      body: registryResponse.body,
      statusCode: registryResponse.status,
      headers: {
        ...CACHE_CONTROL_10_MINUTES_HEADER,
        ...proxiedHeaders
      }
    });
  }
};
exports.getFileHandler = getFileHandler;