"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategoryKeys = getCategoryKeys;
exports.getPartitionFillColor = void 0;
var _charts = require("@elastic/charts");
var _public = require("@kbn/charts-plugin/public");
var _common = require("@kbn/data-plugin/common");
var _types = require("../../../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getCategoryKeys(category) {
  return _common.MultiFieldKey.isInstance(category) ? category.keys.map(String) : `${category}`;
}

/**
 * Get the color of a specific slice/section in Pie,donut,waffle and treemap.
 * These chart type shares the same color assignment mechanism.
 */
const getPieFillColor = (layerIndex, numOfLayers, getColorFn) => (_key, _sortIndex, node) => {
  const path = node[_charts.PATH_KEY];
  // the category used to color the pie/donut is at the third level of the path
  // first two are: small multiple and pie whole center.
  const category = getCategoryKeys(path[2].value);
  const color = getColorFn(category);
  // progressively decrease opacity of base color on each layer.
  return (0, _public.decreaseOpacity)(color, layerIndex + 1, numOfLayers);
};

/**
 * Get the color of a section in a Mosaic chart.
 * This chart has a slight variation in the way color are applied. Mosaic can represent up to 2 layers,
 * described in lens as the horizontal and vertical axes.
 * With a single layer the color is simply applied per each category, with 2 layer, the color is applied only
 * to the category that describe a row, not by column.
 */
const getMosaicFillColor = (layerIndex, numOfLayers, getColorFn) => (_key, _sortIndex, node) => {
  // Special case for 2 layer mosaic where the color is per rows and the columns are not colored
  if (numOfLayers === 2 && layerIndex === 0) {
    // transparent color will fallback to the kibana/context background
    return 'rgba(0,0,0,0)';
  }
  const path = node[_charts.PATH_KEY];

  // the category used to color the pie/donut is at the third level of the `path` when using a single layer mosaic
  // and are at fourth level of `path` when using 2 layer mosaic
  // first two are: small multiple and pie whole center.
  const category = getCategoryKeys(numOfLayers === 2 ? path[3].value : path[2].value);
  return getColorFn(category);
};
const getPartitionFillColor = (chartType, layerIndex, numOfLayers, getColorFn) => {
  return chartType === _types.ChartTypes.MOSAIC ? getMosaicFillColor(layerIndex, numOfLayers, getColorFn) : getPieFillColor(layerIndex, numOfLayers, getColorFn);
};
exports.getPartitionFillColor = getPartitionFillColor;