"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SettingsLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _app_logic = require("../../app_logic");
var _constants = require("../../constants");
var _routes = require("../../routes");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const imageRoute = '/internal/workplace_search/org/settings/upload_images';
const SettingsLogic = exports.SettingsLogic = (0, _kea.kea)({
  actions: {
    onInitializeConnectors: connectors => connectors,
    onOrgNameInputChange: orgNameInputValue => orgNameInputValue,
    setUpdatedName: ({
      organizationName
    }) => organizationName,
    setServerProps: props => props,
    setIcon: icon => icon,
    setStagedIcon: stagedIcon => stagedIcon,
    setLogo: logo => logo,
    setStagedLogo: stagedLogo => stagedLogo,
    setOauthApplication: oauthApplication => oauthApplication,
    setUpdatedOauthApplication: ({
      oauthApplication
    }) => oauthApplication,
    resetSettingsState: () => true,
    initializeSettings: () => true,
    initializeConnectors: () => true,
    updateOrgName: () => true,
    updateOrgLogo: () => true,
    updateOrgIcon: () => true,
    resetOrgLogo: () => true,
    resetOrgIcon: () => true,
    resetButtonLoading: () => true,
    updateOauthApplication: () => true,
    deleteSourceConfig: (serviceType, name) => ({
      serviceType,
      name
    })
  },
  reducers: {
    connectors: [[], {
      onInitializeConnectors: (_, connectors) => (0, _utils.sortByName)(connectors)
    }],
    orgNameInputValue: ['', {
      setServerProps: (_, {
        organizationName
      }) => organizationName,
      onOrgNameInputChange: (_, orgNameInputValue) => orgNameInputValue,
      setUpdatedName: (_, organizationName) => organizationName
    }],
    oauthApplication: [null, {
      setServerProps: (_, {
        oauthApplication
      }) => oauthApplication,
      setOauthApplication: (_, oauthApplication) => oauthApplication,
      setUpdatedOauthApplication: (_, oauthApplication) => oauthApplication
    }],
    dataLoading: [true, {
      setServerProps: () => false,
      onInitializeConnectors: () => false,
      resetSettingsState: () => true
    }],
    logo: [null, {
      setServerProps: (_, {
        logo
      }) => logo,
      setLogo: (_, logo) => logo,
      resetOrgLogo: () => null
    }],
    stagedLogo: [null, {
      setStagedLogo: (_, stagedLogo) => stagedLogo,
      resetOrgLogo: () => null,
      setLogo: () => null
    }],
    icon: [null, {
      setServerProps: (_, {
        icon
      }) => icon,
      setIcon: (_, icon) => icon,
      resetOrgIcon: () => null
    }],
    stagedIcon: [null, {
      setStagedIcon: (_, stagedIcon) => stagedIcon,
      resetOrgIcon: () => null,
      setIcon: () => null
    }],
    logoButtonLoading: [false, {
      updateOrgLogo: () => true,
      setLogo: () => false,
      resetButtonLoading: () => false
    }],
    iconButtonLoading: [false, {
      updateOrgIcon: () => true,
      setIcon: () => false,
      resetButtonLoading: () => false
    }]
  },
  listeners: ({
    actions,
    values
  }) => ({
    initializeSettings: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/workplace_search/org/settings';
      try {
        const response = await http.get(route);
        actions.setServerProps(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    initializeConnectors: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/workplace_search/org/settings/connectors';
      try {
        const response = await http.get(route);
        actions.onInitializeConnectors(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateOrgName: async () => {
      (0, _flash_messages.clearFlashMessages)();
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/workplace_search/org/settings/customize';
      const {
        orgNameInputValue: name
      } = values;
      const body = JSON.stringify({
        name
      });
      try {
        const response = await http.put(route, {
          body
        });
        actions.setUpdatedName(response);
        (0, _flash_messages.flashSuccessToast)(_constants.ORG_UPDATED_MESSAGE);
        _app_logic.AppLogic.actions.setOrgName(name);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateOrgLogo: async () => {
      (0, _flash_messages.clearFlashMessages)();
      const {
        http
      } = _http.HttpLogic.values;
      const {
        stagedLogo: logo
      } = values;
      const body = JSON.stringify({
        logo
      });
      try {
        const response = await http.put(imageRoute, {
          body
        });
        actions.setLogo(response.logo);
        (0, _flash_messages.flashSuccessToast)(_constants.ORG_UPDATED_MESSAGE);
      } catch (e) {
        actions.resetButtonLoading();
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateOrgIcon: async () => {
      (0, _flash_messages.clearFlashMessages)();
      const {
        http
      } = _http.HttpLogic.values;
      const {
        stagedIcon: icon
      } = values;
      const body = JSON.stringify({
        icon
      });
      try {
        const response = await http.put(imageRoute, {
          body
        });
        actions.setIcon(response.icon);
        (0, _flash_messages.flashSuccessToast)(_constants.ORG_UPDATED_MESSAGE);
      } catch (e) {
        actions.resetButtonLoading();
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateOauthApplication: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = '/internal/workplace_search/org/settings/oauth_application';
      const oauthApplication = values.oauthApplication || {};
      const {
        name,
        redirectUri,
        confidential
      } = oauthApplication;
      const body = JSON.stringify({
        oauth_application: {
          name,
          confidential,
          redirect_uri: redirectUri
        }
      });
      (0, _flash_messages.clearFlashMessages)();
      try {
        const response = await http.put(route, {
          body
        });
        actions.setUpdatedOauthApplication(response);
        (0, _flash_messages.flashSuccessToast)(_constants.OAUTH_APP_UPDATED_MESSAGE);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteSourceConfig: async ({
      serviceType,
      name
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const route = `/internal/workplace_search/org/settings/connectors/${serviceType}`;
      try {
        await http.delete(route);
        _kibana.KibanaLogic.values.navigateToUrl(_routes.ORG_SETTINGS_CONNECTORS_PATH);
        (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.settings.configRemoved.message', {
          defaultMessage: 'Successfully removed configuration for {name}.',
          values: {
            name
          }
        }));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    resetSettingsState: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    setStagedLogo: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    setStagedIcon: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    resetOrgLogo: () => {
      actions.updateOrgLogo();
    },
    resetOrgIcon: () => {
      actions.updateOrgIcon();
    }
  })
});