"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticAssistantPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _rxjs = require("rxjs");
var _server = require("@kbn/alerting-plugin/server");
var _server2 = require("@kbn/rule-registry-plugin/server");
var _common = require("@kbn/alerting-plugin/common");
var _event_based_telemetry = require("./lib/telemetry/event_based_telemetry");
var _ai_assistant_service = require("./ai_assistant_service");
var _request_context_factory = require("./routes/request_context_factory");
var _create_event_logger = require("./create_event_logger");
var _constants = require("../common/constants");
var _register_event_log_provider = require("./register_event_log_provider");
var _register_routes = require("./routes/register_routes");
var _app_context = require("./services/app_context");
var _helpers = require("./ai_assistant_service/helpers");
var _definition = require("./lib/attack_discovery/schedules/register_schedule/definition");
var _fields = require("./lib/attack_discovery/schedules/fields");
var _constants2 = require("./lib/attack_discovery/schedules/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ElasticAssistantPlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "assistantService", void 0);
    (0, _defineProperty2.default)(this, "pluginStop$", void 0);
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    this.pluginStop$ = new _rxjs.ReplaySubject(1);
    this.logger = initializerContext.logger.get();
    this.kibanaVersion = initializerContext.env.packageInfo.version;
    this.config = initializerContext.config.get();
  }
  setup(core, plugins) {
    this.logger.debug('elasticAssistant: Setup');
    (0, _register_event_log_provider.registerEventLogProvider)(plugins.eventLog);
    const eventLogger = (0, _create_event_logger.createEventLogger)(plugins.eventLog); // must be created during setup phase

    this.assistantService = new _ai_assistant_service.AIAssistantService({
      logger: this.logger.get('service'),
      ml: plugins.ml,
      taskManager: plugins.taskManager,
      kibanaVersion: this.kibanaVersion,
      elserInferenceId: this.config.elserInferenceId,
      elasticsearchClientPromise: core.getStartServices().then(([{
        elasticsearch
      }]) => elasticsearch.client.asInternalUser),
      soClientPromise: core.getStartServices().then(([{
        savedObjects
      }]) => savedObjects.createInternalRepository()),
      productDocManager: core.getStartServices().then(([_, {
        productDocBase
      }]) => productDocBase.management),
      pluginStop$: this.pluginStop$
    });
    const requestContextFactory = new _request_context_factory.RequestContextFactory({
      logger: this.logger,
      core,
      plugins,
      kibanaVersion: this.kibanaVersion,
      assistantService: this.assistantService
    });
    const router = core.http.createRouter();
    core.http.registerRouteHandlerContext(_constants.PLUGIN_ID, (context, request) => requestContextFactory.create(context, request, plugins.eventLog.getIndexPattern(), eventLogger));
    _event_based_telemetry.events.forEach(eventConfig => core.analytics.registerEventType(eventConfig));
    (0, _register_routes.registerRoutes)(router, this.logger, this.config);

    // The featureFlags service is not available in the core setup, so we need
    // to wait for the start services to be available to read the feature flags.
    // This can take a while, but the plugin setup phase cannot run for a long time.
    // As a workaround, this promise does not block the setup phase.
    const featureFlagDefinitions = [{
      featureFlagName: _elasticAssistantCommon.ATTACK_DISCOVERY_SCHEDULES_ENABLED_FEATURE_FLAG,
      fallbackValue: true,
      fn: assistantAttackDiscoverySchedulingEnabled => {
        if (assistantAttackDiscoverySchedulingEnabled) {
          // Register Attack Discovery Schedule type
          plugins.alerting.registerType((0, _definition.getAttackDiscoveryScheduleType)({
            logger: this.logger,
            publicBaseUrl: core.http.basePath.publicBaseUrl,
            telemetry: core.analytics
          }));
        }
        return !assistantAttackDiscoverySchedulingEnabled; // keep subscription active while the feature flag is disabled
      }
    }, {
      featureFlagName: _elasticAssistantCommon.ATTACK_DISCOVERY_ALERTS_ENABLED_FEATURE_FLAG,
      fallbackValue: true,
      fn: attackDiscoveryAlertsEnabled => {
        let adhocAttackDiscoveryDataClient;
        if (attackDiscoveryAlertsEnabled) {
          // Initialize index for ad-hoc generated attack discoveries
          const {
            ruleDataService
          } = plugins.ruleRegistry;
          const ruleDataServiceOptions = {
            feature: _elasticAssistantCommon.ATTACK_DISCOVERY_SCHEDULES_CONSUMER_ID,
            registrationContext: _constants2.ATTACK_DISCOVERY_ALERTS_CONTEXT,
            dataset: _server2.Dataset.alerts,
            additionalPrefix: '.adhoc',
            componentTemplateRefs: [_server.ECS_COMPONENT_TEMPLATE_NAME],
            componentTemplates: [{
              name: 'mappings',
              mappings: (0, _common.mappingFromFieldMap)(_fields.attackDiscoveryAlertFieldMap)
            }]
          };
          adhocAttackDiscoveryDataClient = ruleDataService.initializeIndex(ruleDataServiceOptions);
        }
        requestContextFactory.setup(adhocAttackDiscoveryDataClient);
        return !attackDiscoveryAlertsEnabled; // keep subscription active while the feature flag is disabled.
      }
    }];
    core.getStartServices().then(([{
      featureFlags
    }]) => this.evaluateFeatureFlags(featureFlagDefinitions, featureFlags)).catch(error => {
      this.logger.error(`error in security assistant plugin setup: ${error}`);
    });
    return {
      actions: plugins.actions,
      getRegisteredFeatures: pluginName => {
        return _app_context.appContextService.getRegisteredFeatures(pluginName);
      },
      getRegisteredTools: pluginName => {
        return _app_context.appContextService.getRegisteredTools(pluginName);
      }
    };
  }
  start(core, plugins) {
    this.logger.debug('elasticAssistant: Started');
    _app_context.appContextService.start({
      logger: this.logger
    });
    (0, _helpers.removeLegacyQuickPrompt)(core.elasticsearch.client.asInternalUser).then(res => {
      if (res !== null && res !== void 0 && res.total) this.logger.info(`Removed ${res.total} legacy quick prompts from AI Assistant`);
    }).catch(() => {});
    return {
      actions: plugins.actions,
      inference: plugins.inference,
      getRegisteredFeatures: pluginName => {
        return _app_context.appContextService.getRegisteredFeatures(pluginName);
      },
      getRegisteredTools: pluginName => {
        return _app_context.appContextService.getRegisteredTools(pluginName);
      },
      registerFeatures: (pluginName, features) => {
        return _app_context.appContextService.registerFeatures(pluginName, features);
      },
      registerTools: (pluginName, tools) => {
        return _app_context.appContextService.registerTools(pluginName, tools);
      }
    };
  }
  stop() {
    _app_context.appContextService.stop();
    this.pluginStop$.next();
    this.pluginStop$.complete();
  }
  evaluateFeatureFlags(featureFlagDefinitions, featureFlags) {
    featureFlagDefinitions.forEach(({
      featureFlagName,
      fallbackValue,
      fn
    }) => {
      featureFlags.getBooleanValue$(featureFlagName, fallbackValue).pipe((0, _rxjs.takeUntil)(this.pluginStop$), (0, _rxjs.exhaustMap)(async enabled => {
        let continueSubscription = true;
        try {
          continueSubscription = await fn(enabled);
        } catch (error) {
          this.logger.error(`Error during setup based on feature flag ${featureFlagName}: ${error}`);
        }
        return continueSubscription;
      }), (0, _rxjs.takeWhile)(continueSubscription => continueSubscription)).subscribe();
    });
  }
}
exports.ElasticAssistantPlugin = ElasticAssistantPlugin;