"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isInferenceEndpointExists = exports.hasDedicatedInferenceEndpointIndexEntries = exports.getInferenceEndpointId = exports.ensureDedicatedInferenceEndpoint = exports.dryRunTrainedModelDeployment = exports.deleteInferenceEndpoint = exports.createInferenceEndpoint = exports.AIAssistantKnowledgeBaseDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _document = require("langchain/document");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _uuid = require("uuid");
var _server = require("@kbn/data-views-plugin/server");
var _lodash = require("lodash");
var _node_utils = require("@kbn/ml-plugin/server/lib/node_utils");
var _inferenceCommon = require("@kbn/inference-common");
var _2 = require("..");
var _create_knowledge_base_entry = require("./create_knowledge_base_entry");
var _transforms = require("./transforms");
var _constants = require("../../routes/knowledge_base/constants");
var _helpers = require("./helpers");
var _utils = require("../../routes/knowledge_base/entries/utils");
var _security_labs_loader = require("../../lib/langchain/content_loaders/security_labs_loader");
var _field_maps_configuration = require("./field_maps_configuration");
var _audit_events = require("./audit_events");
var _find = require("../find");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Params for when creating KbDataClient in Request Context Factory. Useful if needing to modify
 * configuration after initial plugin start
 */

class AIAssistantKnowledgeBaseDataClient extends _2.AIAssistantDataClient {
  constructor(options) {
    super(options);
    (0, _defineProperty2.default)(this, "getProductDocumentationStatus", async () => {
      var _await$this$options$g;
      return (_await$this$options$g = await this.options.getProductDocumentationStatus()) !== null && _await$this$options$g !== void 0 ? _await$this$options$g : 'uninstalled';
    });
    /**
     * Returns whether setup of the Knowledge Base can be performed (essentially an ML features check)
     *
     */
    (0, _defineProperty2.default)(this, "isSetupAvailable", async () => {
      // ML plugin requires request to retrieve capabilities, which are in turn scoped to the user from the request,
      // so we just test the API for a 404 instead to determine if ML is 'available'
      // TODO: expand to include memory check, see https://github.com/elastic/ml-team/issues/1208#issuecomment-2115770318
      try {
        const esClient = await this.options.elasticsearchClientPromise;
        await esClient.ml.getMemoryStats({
          human: true
        });
      } catch (error) {
        return false;
      }
      return true;
    });
    /**
     * Downloads and installs ELSER model if not already installed
     */
    (0, _defineProperty2.default)(this, "installModel", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Installing ELSER model '${elserId}'...`);
      try {
        await this.options.getTrainedModelsProvider().installElasticModel(elserId);
      } catch (error) {
        this.options.logger.error(`Error installing ELSER model '${elserId}':\n${error}`);
      }
    });
    /**
     * Returns whether ELSER is installed/ready to deploy
     *
     * @returns Promise<boolean> indicating whether the model is installed
     */
    (0, _defineProperty2.default)(this, "isModelInstalled", async () => {
      const elserId = await this.options.getElserId();
      this.options.logger.debug(`Checking if ELSER model '${elserId}' is installed...`);
      try {
        var _getResponse$trained_;
        const getResponse = await this.options.getTrainedModelsProvider().getTrainedModels({
          model_id: elserId,
          include: 'definition_status'
        });
        return Boolean((_getResponse$trained_ = getResponse.trained_model_configs[0]) === null || _getResponse$trained_ === void 0 ? void 0 : _getResponse$trained_.fully_defined);
      } catch (error) {
        if (!(0, _helpers.isModelAlreadyExistsError)(error)) {
          this.options.logger.error(`Error checking if ELSER model '${elserId}' is installed:\n${error}`);
        }
        return false;
      }
    });
    (0, _defineProperty2.default)(this, "isInferenceEndpointExists", async () => isInferenceEndpointExists({
      esClient: await this.options.elasticsearchClientPromise,
      logger: this.options.logger,
      getTrainedModelsProvider: this.options.getTrainedModelsProvider
    }));
    (0, _defineProperty2.default)(this, "createInferenceEndpoint", async () => {
      const esClient = await this.options.elasticsearchClientPromise;
      await createInferenceEndpoint({
        elserId: await this.options.getElserId(),
        esClient,
        logger: this.options.logger,
        inferenceId: await getInferenceEndpointId({
          esClient
        }),
        getTrainedModelsProvider: this.options.getTrainedModelsProvider
      });
    });
    /**
     * Downloads and deploys recommended ELSER (if not already), then loads ES|QL docs
     *
     * NOTE: Before automatically installing ELSER in the background, we should perform deployment resource checks
     * Only necessary for ESS, as Serverless can always auto-install if `productTier === complete`
     * See ml-team issue for providing 'dry run' flag to perform these checks: https://github.com/elastic/ml-team/issues/1208
     *
     * @param options
     * @param options.soClient SavedObjectsClientContract for installing ELSER so that ML SO's are in sync
     *
     * @returns Promise<void>
     */
    (0, _defineProperty2.default)(this, "setupKnowledgeBase", async ({
      ignoreSecurityLabs = false
    }) => {
      const elserId = await this.options.getElserId();
      const esClient = await this.options.elasticsearchClientPromise;
      if (this.options.getIsKBSetupInProgress(this.spaceId)) {
        this.options.logger.debug('Knowledge Base setup already in progress');
        return;
      }
      try {
        this.options.logger.debug('Checking if ML nodes are available...');
        const mlNodesCount = await (0, _node_utils.getMlNodeCount)({
          asInternalUser: esClient
        });
        if (mlNodesCount.count === 0 && mlNodesCount.lazyNodeCount === 0) {
          throw new Error('No ML nodes available');
        }
        this.options.logger.debug('Starting Knowledge Base setup...');
        this.options.setIsKBSetupInProgress(this.spaceId, true);

        // Delete legacy ESQL knowledge base docs if they exist, and silence the error if they do not
        try {
          const legacyESQL = await esClient.deleteByQuery({
            index: this.indexTemplateAndPattern.alias,
            query: {
              bool: {
                must: [{
                  terms: {
                    'metadata.kbResource': ['esql', 'unknown']
                  }
                }]
              }
            }
          });
          if ((legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total) != null && (legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total) > 0) {
            this.options.logger.info(`Removed ${legacyESQL === null || legacyESQL === void 0 ? void 0 : legacyESQL.total} ESQL knowledge base docs from knowledge base data stream: ${this.indexTemplateAndPattern.alias}.`);
          }
        } catch (e) {
          this.options.logger.info('No legacy ESQL or Security Labs knowledge base docs to delete');
        }

        // `pt_tiny_elser` is deployed before the KB setup is started, so we don't need to check for it
        if (!this.options.elserInferenceId) {
          /*
          #1 Check if ELSER model is downloaded
          #2 Check if inference endpoint is deployed
          #3 Dry run ELSER model deployment if not already deployed
          #4 Create inference endpoint if not deployed / delete and create inference endpoint if model was not deployed
          #5 Load Security Labs docs
          */
          const isInstalled = await this.isModelInstalled();
          if (!isInstalled) {
            await this.installModel();
            await (0, _pRetry.default)(async () => (await this.isModelInstalled()) ? Promise.resolve() : Promise.reject(new Error('Model not installed')), {
              minTimeout: 30000,
              maxTimeout: 30000,
              retries: 20
            });
            this.options.logger.debug(`ELSER model '${elserId}' successfully installed!`);
          } else {
            this.options.logger.debug(`ELSER model '${elserId}' is already installed`);
          }
          const inferenceId = await getInferenceEndpointId({
            esClient
          });
          const inferenceExists = await isInferenceEndpointExists({
            esClient,
            inferenceEndpointId: inferenceId,
            getTrainedModelsProvider: this.options.getTrainedModelsProvider,
            logger: this.options.logger
          });
          if (!inferenceExists) {
            await createInferenceEndpoint({
              elserId,
              esClient,
              logger: this.options.logger,
              inferenceId,
              getTrainedModelsProvider: this.options.getTrainedModelsProvider
            });
            this.options.logger.debug(`Inference endpoint for ELSER model '${elserId}' successfully deployed!`);
          } else {
            this.options.logger.debug(`Inference endpoint for ELSER model '${elserId}' is already deployed`);
          }
        }
        if (!ignoreSecurityLabs) {
          this.options.logger.debug(`Checking if Knowledge Base docs have been loaded...`);
          const labsDocsLoaded = await this.isSecurityLabsDocsLoaded();
          if (!labsDocsLoaded) {
            var _loadSecurityLabs;
            // Delete any existing Security Labs content
            const securityLabsDocs = await (await this.options.elasticsearchClientPromise).deleteByQuery({
              index: this.indexTemplateAndPattern.alias,
              query: {
                bool: {
                  must: [{
                    terms: {
                      kb_resource: [_constants.SECURITY_LABS_RESOURCE]
                    }
                  }]
                }
              }
            });
            if (securityLabsDocs !== null && securityLabsDocs !== void 0 && securityLabsDocs.total) {
              this.options.logger.info(`Removed ${securityLabsDocs === null || securityLabsDocs === void 0 ? void 0 : securityLabsDocs.total} Security Labs knowledge base docs from knowledge base data stream: ${this.indexTemplateAndPattern.alias}.`);
            }
            this.options.logger.debug(`Loading Security Labs KB docs...`);
            void ((_loadSecurityLabs = (0, _security_labs_loader.loadSecurityLabs)(this, this.options.logger)) === null || _loadSecurityLabs === void 0 ? void 0 : _loadSecurityLabs.then(() => {
              this.options.setIsKBSetupInProgress(this.spaceId, false);
            }));
          } else {
            this.options.logger.debug(`Security Labs Knowledge Base docs already loaded!`);
          }
        }

        // If loading security labs, we need to wait for the docs to be loaded
        if (ignoreSecurityLabs) {
          this.options.setIsKBSetupInProgress(this.spaceId, false);
        }
      } catch (e) {
        this.options.setIsKBSetupInProgress(this.spaceId, false);
        this.options.logger.error(`Error setting up Knowledge Base: ${e.message}`);
        throw new Error(`Error setting up Knowledge Base: ${e.message}`);
      }
    });
    // TODO make this function private
    // no telemetry, no audit logs
    /**
     * Adds LangChain Documents to the knowledge base
     *
     * @param {Array<Document<Metadata>>} documents - LangChain Documents to add to the knowledge base
     * @param global whether these entries should be added globally, i.e. empty users[]
     */
    (0, _defineProperty2.default)(this, "addKnowledgeBaseDocuments", async ({
      documents,
      global = false
    }) => {
      var _created$data$hits$hi;
      const writer = await this.getWriter();
      const changedAt = new Date().toISOString();
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      if (global && !this.options.manageGlobalKnowledgeBaseAIAssistant) {
        throw new Error('User lacks privileges to create global knowledge base entries');
      }
      const {
        errors,
        docs_created: docsCreated
      } = await writer.bulk({
        documentsToCreate: documents.map(doc => {
          var _doc$metadata$kbResou, _doc$metadata$require, _doc$metadata$source;
          // v1 schema has metadata nested in a `metadata` object
          return (0, _create_knowledge_base_entry.transformToCreateSchema)({
            createdAt: changedAt,
            spaceId: this.spaceId,
            user: authenticatedUser,
            entry: {
              type: _elasticAssistantCommon.DocumentEntryType.value,
              name: 'unknown',
              text: doc.pageContent,
              kbResource: (_doc$metadata$kbResou = doc.metadata.kbResource) !== null && _doc$metadata$kbResou !== void 0 ? _doc$metadata$kbResou : 'unknown',
              required: (_doc$metadata$require = doc.metadata.required) !== null && _doc$metadata$require !== void 0 ? _doc$metadata$require : false,
              source: (_doc$metadata$source = doc.metadata.source) !== null && _doc$metadata$source !== void 0 ? _doc$metadata$source : 'unknown',
              global
            }
          });
        }),
        authenticatedUser
      });
      const created = docsCreated.length > 0 ? await this.findDocuments({
        page: 1,
        perPage: 10000,
        filter: docsCreated.map(c => `_id:${c}`).join(' OR ')
      }) : undefined;
      // Intentionally no telemetry here - this path only used to install security docs
      // Plans to make this function private in a different PR so no user entry ever is created in this path
      this.options.logger.debug(`created: ${(_created$data$hits$hi = created === null || created === void 0 ? void 0 : created.data.hits.hits.length) !== null && _created$data$hits$hi !== void 0 ? _created$data$hits$hi : '0'}`);
      this.options.logger.debug(() => `errors: ${JSON.stringify(errors, null, 2)}`);
      return created !== null && created !== void 0 && created.data ? (0, _transforms.transformESSearchToKnowledgeBaseEntry)(created === null || created === void 0 ? void 0 : created.data) : [];
    });
    /**
     * Returns if user's KB docs exists
     */
    (0, _defineProperty2.default)(this, "isUserDataExists", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      try {
        var _result$hits;
        const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
          kbResource: _constants.USER_RESOURCE,
          required: false,
          user
        });
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 0,
          query: vectorSearchQuery,
          track_total_hits: true
        });
        return !!((_result$hits = result.hits) === null || _result$hits === void 0 ? void 0 : _result$hits.total).value;
      } catch (e) {
        this.options.logger.debug(`Error checking if user's KB docs exist: ${e.message}`);
        return false;
      }
    });
    /**
     * Returns loaded Security Labs KB docs count
     */
    (0, _defineProperty2.default)(this, "getLoadedSecurityLabsDocsCount", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      try {
        var _result$hits2;
        const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
          kbResource: _constants.SECURITY_LABS_RESOURCE,
          required: false,
          user
        });
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 0,
          query: vectorSearchQuery,
          track_total_hits: true
        });
        return ((_result$hits2 = result.hits) === null || _result$hits2 === void 0 ? void 0 : _result$hits2.total).value;
      } catch (e) {
        this.options.logger.info(`Error checking if Security Labs docs are loaded: ${e.message}`);
        return 0;
      }
    });
    /**
     * Returns if allSecurity Labs KB docs have been loaded
     */
    (0, _defineProperty2.default)(this, "isSecurityLabsDocsLoaded", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      try {
        const expectedDocsCount = await (0, _security_labs_loader.getSecurityLabsDocsCount)({
          logger: this.options.logger
        });
        const existingDocs = await this.getLoadedSecurityLabsDocsCount();
        if (existingDocs !== expectedDocsCount) {
          this.options.logger.debug(`Security Labs docs are not loaded, existing docs: ${existingDocs}, expected docs: ${expectedDocsCount}`);
        }
        return existingDocs === expectedDocsCount;
      } catch (e) {
        this.options.logger.info(`Error checking if Security Labs docs are loaded: ${e.message}`);
        return false;
      }
    });
    /**
     * Performs similarity search to retrieve LangChain Documents from the knowledge base
     */
    (0, _defineProperty2.default)(this, "getKnowledgeBaseDocumentEntries", async ({
      filter,
      kbResource,
      query,
      required
    }) => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      const esClient = await this.options.elasticsearchClientPromise;
      const vectorSearchQuery = (0, _helpers.getKBVectorSearchQuery)({
        filter,
        kbResource,
        query,
        required,
        user
      });
      try {
        const result = await esClient.search({
          index: this.indexTemplateAndPattern.alias,
          size: 10,
          query: vectorSearchQuery
        });
        const results = result.hits.hits.map(hit => {
          var _hit$_source, _hit$_source2, _hit$_source3, _hit$_source4, _hit$_source$text, _hit$_source5;
          const metadata = {
            name: hit === null || hit === void 0 ? void 0 : (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.name,
            index: hit === null || hit === void 0 ? void 0 : hit._index,
            source: hit === null || hit === void 0 ? void 0 : (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.source,
            required: hit === null || hit === void 0 ? void 0 : (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : _hit$_source3.required,
            kbResource: hit === null || hit === void 0 ? void 0 : (_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : _hit$_source4.kb_resource
          };
          return new _document.Document({
            id: hit === null || hit === void 0 ? void 0 : hit._id,
            pageContent: (_hit$_source$text = hit === null || hit === void 0 ? void 0 : (_hit$_source5 = hit._source) === null || _hit$_source5 === void 0 ? void 0 : _hit$_source5.text) !== null && _hit$_source$text !== void 0 ? _hit$_source$text : '',
            metadata
          });
        });
        this.options.logger.debug(() => `getKnowledgeBaseDocuments() - Similarity Search Query:\n ${JSON.stringify(vectorSearchQuery)}`);
        this.options.logger.debug(() => `getKnowledgeBaseDocuments() - Similarity Search returned [${JSON.stringify(results.length)}] results`);
        return results;
      } catch (e) {
        this.options.logger.error(`Error performing KB Similarity Search: ${e.message}`);
        return [];
      }
    });
    /**
     * Returns all global and current user's private `required` document entries.
     */
    (0, _defineProperty2.default)(this, "getRequiredKnowledgeBaseDocumentEntries", async () => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      try {
        const userFilter = (0, _utils.getKBUserFilter)(user);
        const results = await this.findDocuments({
          // Note: This is a magic number to set some upward bound as to not blow the context with too
          // many historical KB entries. Ideally we'd query for all and token trim.
          perPage: 100,
          page: 1,
          sortField: 'created_at',
          sortOrder: 'asc',
          filter: `${userFilter} AND type:document AND kb_resource:user AND required:true`
        });
        this.options.logger.debug(`kbDataClient.getRequiredKnowledgeBaseDocumentEntries() - results:\n${JSON.stringify(results)}`);
        if (results) {
          return (0, _transforms.transformESSearchToKnowledgeBaseEntry)(results.data);
        }
      } catch (e) {
        this.options.logger.error(`kbDataClient.getRequiredKnowledgeBaseDocumentEntries() - Failed to fetch DocumentEntries`);
        return [];
      }
      return [];
    });
    /**
     * Creates a new Knowledge Base Entry.
     *
     * @param knowledgeBaseEntry
     * @param global
     */
    (0, _defineProperty2.default)(this, "createKnowledgeBaseEntry", async ({
      auditLogger,
      knowledgeBaseEntry,
      telemetry
    }) => {
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      if ((0, _utils.isGlobalEntry)(knowledgeBaseEntry) && !this.options.manageGlobalKnowledgeBaseAIAssistant) {
        throw new Error('User lacks privileges to create global knowledge base entries');
      }
      this.options.logger.debug(() => `Creating Knowledge Base Entry:\n ${JSON.stringify(knowledgeBaseEntry, null, 2)}`);
      this.options.logger.debug(`kbIndex: ${this.indexTemplateAndPattern.alias}`);
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _create_knowledge_base_entry.createKnowledgeBaseEntry)({
        auditLogger,
        esClient,
        knowledgeBaseIndex: this.indexTemplateAndPattern.alias,
        logger: this.options.logger,
        spaceId: this.spaceId,
        user: authenticatedUser,
        knowledgeBaseEntry,
        telemetry
      });
    });
    /**
     * Updates a Knowledge Base Entry.
     *
     * @param auditLogger
     * @param knowledgeBaseEntryId
     */
    (0, _defineProperty2.default)(this, "updateKnowledgeBaseEntry", async ({
      auditLogger,
      knowledgeBaseEntry,
      telemetry
    }) => {
      var _transformESToKnowled;
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      await (0, _utils.validateDocumentsModification)(this, authenticatedUser, [knowledgeBaseEntry.id], 'update');
      this.options.logger.debug(() => `Updating Knowledge Base Entry:\n ${JSON.stringify(knowledgeBaseEntry, null, 2)}`);
      this.options.logger.debug(`kbIndex: ${this.indexTemplateAndPattern.alias}`);
      const writer = await this.getWriter();
      const changedAt = new Date().toISOString();
      const {
        errors,
        docs_updated: docsUpdated
      } = await writer.bulk({
        documentsToUpdate: [(0, _create_knowledge_base_entry.transformToUpdateSchema)({
          user: authenticatedUser,
          updatedAt: changedAt,
          entry: knowledgeBaseEntry
        })],
        getUpdateScript: entry => (0, _create_knowledge_base_entry.getUpdateScript)({
          entry
        }),
        authenticatedUser
      });

      // @ts-ignore-next-line TS2322
      const updatedEntry = (_transformESToKnowled = (0, _transforms.transformESToKnowledgeBase)(docsUpdated)) === null || _transformESToKnowled === void 0 ? void 0 : _transformESToKnowled[0];
      if (updatedEntry) {
        auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
          action: _audit_events.KnowledgeBaseAuditAction.UPDATE,
          id: updatedEntry.id,
          name: updatedEntry.name,
          outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
        }));
      }
      return {
        errors,
        updatedEntry
      };
    });
    /**
     * Deletes a new Knowledge Base Entry.
     *
     * @param auditLogger
     * @param knowledgeBaseEntryId
     */
    (0, _defineProperty2.default)(this, "deleteKnowledgeBaseEntry", async ({
      auditLogger,
      knowledgeBaseEntryId
    }) => {
      const authenticatedUser = this.options.currentUser;
      if (authenticatedUser == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      await (0, _utils.validateDocumentsModification)(this, authenticatedUser, [knowledgeBaseEntryId], 'delete');
      this.options.logger.debug(() => `Deleting Knowledge Base Entry:\n ID: ${JSON.stringify(knowledgeBaseEntryId, null, 2)}`);
      this.options.logger.debug(`kbIndex: ${this.indexTemplateAndPattern.alias}`);
      const writer = await this.getWriter();
      const {
        errors,
        docs_deleted: docsDeleted
      } = await writer.bulk({
        documentsToDelete: [knowledgeBaseEntryId],
        authenticatedUser
      });
      if (docsDeleted.length) {
        docsDeleted.forEach(docsDeletedId => {
          auditLogger === null || auditLogger === void 0 ? void 0 : auditLogger.log((0, _audit_events.knowledgeBaseAuditEvent)({
            action: _audit_events.KnowledgeBaseAuditAction.DELETE,
            id: docsDeletedId,
            outcome: _audit_events.AUDIT_OUTCOME.SUCCESS
          }));
        });
      }
      return {
        errors,
        docsDeleted
      };
    });
    /**
     * Returns AssistantTools for any 'relevant' KB IndexEntries that exist in the knowledge base.
     *
     * Note: Accepts esClient so retrieval can be scoped to the current user as esClient on kbDataClient
     * is scoped to system user.
     */
    (0, _defineProperty2.default)(this, "getAssistantTools", async ({
      contentReferencesStore,
      esClient
    }) => {
      const user = this.options.currentUser;
      if (user == null) {
        throw new Error('Authenticated user not found! Ensure kbDataClient was initialized from a request.');
      }
      try {
        const userFilter = (0, _utils.getKBUserFilter)(user);
        const results = await this.findDocuments({
          // Note: This is a magic number to set some upward bound as to not blow the context with too
          // many registered tools. As discussed in review, this will initially be mitigated by caps on
          // the IndexEntries field lengths, context trimming at the graph layer (before compilation),
          // and eventually some sort of tool discovery sub-graph or generic retriever to scale tool usage.
          perPage: 23,
          page: 1,
          sortField: 'created_at',
          sortOrder: 'asc',
          filter: `${userFilter} AND type:index`
        });
        this.options.logger.debug(`kbDataClient.getAssistantTools() - results:\n${JSON.stringify(results, null, 2)}`);
        if (results) {
          const entries = (0, _transforms.transformESSearchToKnowledgeBaseEntry)(results.data);
          const indexPatternFetcher = new _server.IndexPatternsFetcher(esClient);
          const existingIndices = await indexPatternFetcher.getExistingIndices((0, _lodash.map)(entries, 'index'));
          return entries
          // Filter out any IndexEntries that don't have an existing index
          .filter(entry => existingIndices.includes(entry.index)).map(indexEntry => {
            return (0, _helpers.getStructuredToolForIndexEntry)({
              indexEntry,
              esClient,
              logger: this.options.logger,
              contentReferencesStore
            });
          });
        }
      } catch (e) {
        this.options.logger.error(`kbDataClient.getAssistantTools() - Failed to fetch IndexEntries`);
        return [];
      }
      return [];
    });
    this.options = options;
  }
  get isSetupInProgress() {
    return this.options.getIsKBSetupInProgress(this.spaceId);
  }
}
exports.AIAssistantKnowledgeBaseDataClient = AIAssistantKnowledgeBaseDataClient;
const getInferenceEndpointId = async ({
  esClient
}) => {
  try {
    const elasticsearchInference = await esClient.inference.get({
      inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
      task_type: 'sparse_embedding'
    });
    if (elasticsearchInference) {
      return _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID;
    }
  } catch (_) {
    /* empty */
  }

  // Fallback to the default inference endpoint
  return _inferenceCommon.defaultInferenceEndpoints.ELSER;
};

/**
 * Checks if the inference endpoint is deployed and allocated in Elasticsearch
 *
 * @returns Promise<boolean> indicating whether the model is deployed
 */
exports.getInferenceEndpointId = getInferenceEndpointId;
const isInferenceEndpointExists = async ({
  esClient,
  inferenceEndpointId,
  getTrainedModelsProvider,
  logger
}) => {
  const inferenceId = inferenceEndpointId || (await getInferenceEndpointId({
    esClient
  }));
  try {
    var _response$endpoints, _response$endpoints$, _response$endpoints$$, _getResponse$trained_2;
    const response = await esClient.inference.get({
      inference_id: inferenceId,
      task_type: 'sparse_embedding'
    });
    if (!((_response$endpoints = response.endpoints) !== null && _response$endpoints !== void 0 && (_response$endpoints$ = _response$endpoints[0]) !== null && _response$endpoints$ !== void 0 && (_response$endpoints$$ = _response$endpoints$.service_settings) !== null && _response$endpoints$$ !== void 0 && _response$endpoints$$.model_id)) {
      return false;
    }
    let getResponse;
    try {
      var _response$endpoints2, _response$endpoints2$, _response$endpoints2$2;
      getResponse = await getTrainedModelsProvider().getTrainedModelsStats({
        model_id: (_response$endpoints2 = response.endpoints) === null || _response$endpoints2 === void 0 ? void 0 : (_response$endpoints2$ = _response$endpoints2[0]) === null || _response$endpoints2$ === void 0 ? void 0 : (_response$endpoints2$2 = _response$endpoints2$.service_settings) === null || _response$endpoints2$2 === void 0 ? void 0 : _response$endpoints2$2.model_id
      });
    } catch (e) {
      return false;
    }

    // For standardized way of checking deployment status see: https://github.com/elastic/elasticsearch/issues/106986
    const isReadyESS = stats => {
      var _stats$deployment_sta, _stats$deployment_sta2, _stats$deployment_sta3;
      return ((_stats$deployment_sta = stats.deployment_stats) === null || _stats$deployment_sta === void 0 ? void 0 : _stats$deployment_sta.state) === 'started' && ((_stats$deployment_sta2 = stats.deployment_stats) === null || _stats$deployment_sta2 === void 0 ? void 0 : (_stats$deployment_sta3 = _stats$deployment_sta2.allocation_status) === null || _stats$deployment_sta3 === void 0 ? void 0 : _stats$deployment_sta3.state) === 'fully_allocated';
    };
    const isReadyServerless = stats => {
      var _stats$deployment_sta4, _stats$deployment_sta5;
      return (_stats$deployment_sta4 = stats.deployment_stats) === null || _stats$deployment_sta4 === void 0 ? void 0 : (_stats$deployment_sta5 = _stats$deployment_sta4.nodes) === null || _stats$deployment_sta5 === void 0 ? void 0 : _stats$deployment_sta5.some(node => node.routing_state.routing_state === 'started');
    };
    return !!((_getResponse$trained_2 = getResponse.trained_model_stats.filter(stats => {
      var _stats$deployment_sta6;
      return ((_stats$deployment_sta6 = stats.deployment_stats) === null || _stats$deployment_sta6 === void 0 ? void 0 : _stats$deployment_sta6.deployment_id) === inferenceId;
    })) !== null && _getResponse$trained_2 !== void 0 && _getResponse$trained_2.some(stats => isReadyESS(stats) || isReadyServerless(stats)));
  } catch (error) {
    logger.debug(`Error checking if Inference endpoint ${inferenceId} exists: ${error}`);
    return false;
  }
};
exports.isInferenceEndpointExists = isInferenceEndpointExists;
const hasDedicatedInferenceEndpointIndexEntries = async ({
  esClient,
  index,
  logger
}) => {
  try {
    // Track if we've already created an inference endpoint to ensure we only create it once
    let inferenceEndpointUsed = false;

    // Pull knowledge base entries of type "index"
    const results = await (0, _find.findDocuments)({
      perPage: 100,
      page: 1,
      filter: 'type:index',
      esClient,
      logger,
      index
    });
    if (!results || !results.data.hits.hits.length) {
      logger.debug('No index type knowledge base entries found');
      return false;
    }
    const indexEntries = (0, _transforms.transformESSearchToKnowledgeBaseEntry)(results.data);
    for (const entry of indexEntries) {
      // Skip if we've already created the inference endpoint
      if (inferenceEndpointUsed) {
        break;
      }
      try {
        // Get the specific field mapping directly from Elasticsearch
        const fieldMappingResponse = await esClient.indices.getFieldMapping({
          index: entry.index,
          fields: entry.field,
          include_defaults: true
        });

        // Check each index for the field with inference_id
        for (const indexName of Object.keys(fieldMappingResponse)) {
          // Skip if we've already created the inference endpoint
          if (inferenceEndpointUsed) {
            break;
          }
          const mappings = fieldMappingResponse[indexName].mappings || {};

          // Check if the field exists and has mapping information
          if (mappings[entry.field] && mappings[entry.field].mapping) {
            // The field mapping structure is nested under the field name and then 'mapping'
            const fieldMapping = mappings[entry.field].mapping[entry.field];

            // Check if it's a semantic_text field with the specific inference_id
            if (fieldMapping && fieldMapping.type === 'semantic_text' && fieldMapping.inference_id === _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID) {
              inferenceEndpointUsed = true;

              // No need to check other fields or indices
              break;
            }
          }
        }
      } catch (error) {
        logger.error(`Error checking field mappings for index ${entry.index}: ${error.message}`);
      }
    }
    return inferenceEndpointUsed;
  } catch (error) {
    return false;
  }
};
exports.hasDedicatedInferenceEndpointIndexEntries = hasDedicatedInferenceEndpointIndexEntries;
const dryRunTrainedModelDeployment = async ({
  esClient,
  elserId,
  logger,
  getTrainedModelsProvider
}) => {
  try {
    const deploymentId = (0, _uuid.v4)();
    // As there is no better way to check if the model is deployed, we try to start the model
    // deployment and throw an error if it fails
    const dryRunId = await esClient.ml.startTrainedModelDeployment({
      model_id: elserId,
      deployment_id: deploymentId,
      wait_for: 'fully_allocated'
    });
    logger.debug(`Dry run for ELSER model '${elserId}' successfully deployed!`);
    await getTrainedModelsProvider().stopTrainedModelDeployment({
      model_id: elserId,
      deployment_id: dryRunId.assignment.task_parameters.deployment_id
    });
    logger.debug(`Dry run for ELSER model '${elserId}' successfully stopped!`);
  } catch (e) {
    logger.error(`Dry run error starting trained model deployment: ${e.message}`);
    throw new Error(`${e.message}`);
  }
};
exports.dryRunTrainedModelDeployment = dryRunTrainedModelDeployment;
const deleteInferenceEndpoint = async ({
  esClient,
  logger
}) => {
  try {
    await esClient.inference.delete({
      inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
      // it's being used in the mapping so we need to force delete
      force: true
    });
    logger.debug(`Deleted existing inference endpoint ${_field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID} for ELSER model`);
  } catch (error) {
    logger.debug(`Error deleting inference endpoint ${_field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID} for ELSER model:\n${error}`);
  }
};
exports.deleteInferenceEndpoint = deleteInferenceEndpoint;
const createInferenceEndpoint = async ({
  elserId,
  logger,
  esClient,
  inferenceId,
  getTrainedModelsProvider
}) => {
  if (inferenceId === _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID) {
    await deleteInferenceEndpoint({
      esClient,
      logger
    });
    await (0, _pRetry.default)(async () => dryRunTrainedModelDeployment({
      elserId,
      esClient,
      logger,
      getTrainedModelsProvider
    }), {
      minTimeout: 10000,
      maxTimeout: 10000,
      retries: 1
    });
    try {
      await esClient.inference.put({
        task_type: 'sparse_embedding',
        inference_id: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
        inference_config: {
          service: 'elasticsearch',
          service_settings: {
            adaptive_allocations: {
              enabled: true,
              min_number_of_allocations: 0,
              max_number_of_allocations: 8
            },
            num_threads: 1,
            model_id: elserId
          },
          task_settings: {}
        }
      });

      // await for the model to be deployed
      const inferenceEndpointExists = await isInferenceEndpointExists({
        esClient,
        inferenceEndpointId: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
        getTrainedModelsProvider,
        logger
      });
      if (!inferenceEndpointExists) {
        throw new Error(`Inference endpoint for ELSER model '${elserId}' was not deployed successfully`);
      }
    } catch (error) {
      logger.error(`Error creating inference endpoint for ELSER model '${elserId}':\n${error}`);
      throw new Error(`Error creating inference endpoint for ELSER model '${elserId}':\n${error}`);
    }
  } else {
    await dryRunTrainedModelDeployment({
      elserId,
      esClient,
      logger,
      getTrainedModelsProvider
    });
  }
};
exports.createInferenceEndpoint = createInferenceEndpoint;
const ensureDedicatedInferenceEndpoint = async ({
  elserId,
  esClient,
  logger,
  index,
  getTrainedModelsProvider
}) => {
  try {
    const isEndpointUsed = await hasDedicatedInferenceEndpointIndexEntries({
      esClient,
      index,
      logger
    });
    const inferenceEndpointExists = await isInferenceEndpointExists({
      esClient,
      inferenceEndpointId: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
      getTrainedModelsProvider,
      logger
    });
    if (!isEndpointUsed) {
      if (inferenceEndpointExists) {
        await deleteInferenceEndpoint({
          esClient,
          logger
        });
      }
      return;
    }
    if (inferenceEndpointExists) {
      logger.debug(`Inference endpoint ${_field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID} already exists.`);
      return;
    }
    await createInferenceEndpoint({
      elserId,
      esClient,
      logger,
      inferenceId: _field_maps_configuration.ASSISTANT_ELSER_INFERENCE_ID,
      getTrainedModelsProvider
    });
  } catch (error) {
    logger.error(`Error in ensureDedicatedInferenceEndpoint: ${error.message}`);
  }
};
exports.ensureDedicatedInferenceEndpoint = ensureDedicatedInferenceEndpoint;