"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AppMenuRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class AppMenuRegistry {
  constructor(primaryAndSecondaryActions) {
    (0, _defineProperty2.default)(this, "appMenuItems", void 0);
    /**
     * As custom actions can be registered under a submenu from both root and data source profiles, we need to keep track of them separately.
     * Otherwise, it would be less predictable. For example, we would override/reset the actions from the data source profile with the ones from the root profile.
     * @internal
     */
    (0, _defineProperty2.default)(this, "customSubmenuItemsBySubmenuId", void 0);
    this.appMenuItems = assignOrderToActions(primaryAndSecondaryActions);
    this.customSubmenuItemsBySubmenuId = new Map();
  }
  isActionRegistered(appMenuItemId) {
    return this.appMenuItems.some(item => {
      if (item.id === appMenuItemId) {
        return true;
      }
      if (isAppMenuActionSubmenu(item)) {
        return item.actions.some(submenuItem => submenuItem.id === appMenuItemId);
      }
      return false;
    }) || [...this.customSubmenuItemsBySubmenuId.values()].some(submenuItems => submenuItems.some(item => item.id === appMenuItemId));
  }

  /**
   * Register a custom action to the app menu. It can be a simple action or a submenu with more actions and horizontal rules.
   * Note: Only 2 top level custom actions are allowed to be rendered in the app menu. The rest will be ignored.
   * A custom action can also open a flyout or a modal. For that, return your custom react node from action's `onClick` event and call `onFinishAction` when you're done.
   * @param appMenuItem
   */
  registerCustomAction(appMenuItem) {
    this.appMenuItems = [...this.appMenuItems.filter(
    // prevent duplicates
    item => !(item.id === appMenuItem.id && item.type === _types.AppMenuActionType.custom)), appMenuItem];
  }

  /**
   * Register a custom action under a submenu. It can be an action or a horizontal rule.
   * Any number of submenu actions can be registered and rendered.
   * You can also extend an existing submenu with more actions. For example, AppMenuActionType.alerts.
   * `order` property is optional and can be used to control the order of actions in the submenu.
   * @param submenuId
   * @param appMenuItem
   */
  registerCustomActionUnderSubmenu(submenuId, appMenuItem) {
    var _this$customSubmenuIt;
    this.customSubmenuItemsBySubmenuId.set(submenuId, [...((_this$customSubmenuIt = this.customSubmenuItemsBySubmenuId.get(submenuId)) !== null && _this$customSubmenuIt !== void 0 ? _this$customSubmenuIt : []).filter(
    // prevent duplicates and allow overrides
    item => item.id !== appMenuItem.id), appMenuItem]);
  }
  getSortedItemsForType(type) {
    let actions = this.appMenuItems.filter(item => item.type === type);
    if (type === _types.AppMenuActionType.custom && actions.length > AppMenuRegistry.CUSTOM_ITEMS_LIMIT) {
      // apply the limitation on how many custom items can be shown
      actions = actions.slice(0, AppMenuRegistry.CUSTOM_ITEMS_LIMIT);
    }

    // enrich submenus with custom actions
    if (type === _types.AppMenuActionType.secondary || type === _types.AppMenuActionType.custom) {
      [...this.customSubmenuItemsBySubmenuId.entries()].forEach(([submenuId, customActions]) => {
        actions = actions.map(item => {
          if (item.id === submenuId && isAppMenuActionSubmenu(item)) {
            return extendSubmenuWithCustomActions(item, customActions);
          }
          return item;
        });
      });
    }
    return sortAppMenuItemsByOrder(actions);
  }

  /**
   * Get the resulting app menu items sorted by type and order.
   */
  getSortedItems() {
    const primaryItems = this.getSortedItemsForType(_types.AppMenuActionType.primary);
    const secondaryItems = this.getSortedItemsForType(_types.AppMenuActionType.secondary);
    const customItems = this.getSortedItemsForType(_types.AppMenuActionType.custom);
    return [...customItems, ...secondaryItems, ...primaryItems].filter(item => !isAppMenuActionSubmenu(item) || item.actions.length > 0);
  }
}
exports.AppMenuRegistry = AppMenuRegistry;
(0, _defineProperty2.default)(AppMenuRegistry, "CUSTOM_ITEMS_LIMIT", 2);
function isAppMenuActionSubmenu(appMenuItem) {
  return 'actions' in appMenuItem && Array.isArray(appMenuItem.actions);
}
const FALLBACK_ORDER = Number.MAX_SAFE_INTEGER;
function sortByOrder(a, b) {
  var _a$order, _b$order;
  return ((_a$order = a.order) !== null && _a$order !== void 0 ? _a$order : FALLBACK_ORDER) - ((_b$order = b.order) !== null && _b$order !== void 0 ? _b$order : FALLBACK_ORDER);
}
function getAppMenuSubmenuWithSortedItemsByOrder(appMenuItem) {
  return {
    ...appMenuItem,
    actions: [...appMenuItem.actions].sort(sortByOrder)
  };
}
function sortAppMenuItemsByOrder(appMenuItems) {
  const sortedAppMenuItems = [...appMenuItems].sort(sortByOrder);
  return sortedAppMenuItems.map(appMenuItem => {
    if (isAppMenuActionSubmenu(appMenuItem)) {
      return getAppMenuSubmenuWithSortedItemsByOrder(appMenuItem);
    }
    return appMenuItem;
  });
}
function getAppMenuSubmenuWithAssignedOrder(appMenuItem, order) {
  var _appMenuItem$order;
  let orderInSubmenu = 0;
  const actionsWithOrder = appMenuItem.actions.map(action => {
    var _action$order;
    orderInSubmenu = orderInSubmenu + 100;
    return {
      ...action,
      order: (_action$order = action.order) !== null && _action$order !== void 0 ? _action$order : orderInSubmenu
    };
  });
  return {
    ...appMenuItem,
    order: (_appMenuItem$order = appMenuItem.order) !== null && _appMenuItem$order !== void 0 ? _appMenuItem$order : order,
    actions: actionsWithOrder
  };
}
function extendSubmenuWithCustomActions(appMenuItem, customActions) {
  const customActionsIds = new Set(customActions.map(action => action.id));
  return {
    ...appMenuItem,
    actions: [...appMenuItem.actions.filter(item => !customActionsIds.has(item.id)),
    // allow to override secondary actions with custom ones
    ...customActions]
  };
}

/**
 * All primary and secondary actions by default get order 100, 200, 300,... assigned to them.
 * Same for actions under a submenu.
 * @param appMenuItems
 */
function assignOrderToActions(appMenuItems) {
  let order = 0;
  return appMenuItems.map(appMenuItem => {
    var _appMenuItem$order2;
    order = order + 100;
    if (isAppMenuActionSubmenu(appMenuItem)) {
      return getAppMenuSubmenuWithAssignedOrder(appMenuItem, order);
    }
    return {
      ...appMenuItem,
      order: (_appMenuItem$order2 = appMenuItem.order) !== null && _appMenuItem$order2 !== void 0 ? _appMenuItem$order2 : order
    };
  });
}