"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDefaultProfileState = void 0;
var _lodash = require("lodash");
var _context_awareness = require("../../../../context_awareness");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getDefaultProfileState = ({
  profilesManager,
  resetDefaultProfileState,
  dataView
}) => {
  const defaultState = getDefaultState(profilesManager, dataView);
  return {
    /**
     * Returns state that should be updated before data fetching occurs,
     * for example state used as part of the data fetching process
     * @returns The state to reset to before fetching data
     */
    getPreFetchState: () => {
      const stateUpdate = {};
      if (resetDefaultProfileState.breakdownField && defaultState.breakdownField !== undefined && dataView.fields.getByName(defaultState.breakdownField)) {
        stateUpdate.breakdownField = defaultState.breakdownField;
      }
      if (resetDefaultProfileState.hideChart && defaultState.hideChart !== undefined) {
        stateUpdate.hideChart = defaultState.hideChart;
      }
      return Object.keys(stateUpdate).length ? stateUpdate : undefined;
    },
    /**
     * Returns state that should be updated after data fetching occurs,
     * for example state used to modify the UI after receiving data
     * @returns The state to reset to after fetching data
     */
    getPostFetchState: ({
      defaultColumns,
      esqlQueryColumns
    }) => {
      const stateUpdate = {};
      if (resetDefaultProfileState.columns) {
        var _defaultState$columns;
        const mappedDefaultColumns = defaultColumns.map(name => ({
          name
        }));
        const isValidColumn = getIsValidColumn(dataView, esqlQueryColumns);
        const validColumns = (0, _lodash.uniqBy)((_defaultState$columns = defaultState.columns) === null || _defaultState$columns === void 0 ? void 0 : _defaultState$columns.concat(mappedDefaultColumns).filter(isValidColumn), 'name');
        if (validColumns !== null && validColumns !== void 0 && validColumns.length) {
          const hasAutoWidthColumn = validColumns.some(({
            width
          }) => !width);
          const columns = validColumns.reduce((acc, {
            name,
            width
          }, index) => {
            // Ensure there's at least one auto width column so the columns fill the grid
            const skipColumnWidth = !hasAutoWidthColumn && index === validColumns.length - 1;
            return width && !skipColumnWidth ? {
              ...acc,
              [name]: {
                width
              }
            } : acc;
          }, undefined);
          stateUpdate.grid = columns ? {
            columns
          } : undefined;
          stateUpdate.columns = validColumns.map(({
            name
          }) => name);
        }
      }
      if (resetDefaultProfileState.rowHeight && defaultState.rowHeight !== undefined) {
        stateUpdate.rowHeight = defaultState.rowHeight;
      }
      return Object.keys(stateUpdate).length ? stateUpdate : undefined;
    }
  };
};
exports.getDefaultProfileState = getDefaultProfileState;
const getDefaultState = (profilesManager, dataView) => {
  const getDefaultAppState = (0, _context_awareness.getMergedAccessor)(profilesManager.getProfiles(), 'getDefaultAppState', () => ({}));
  return getDefaultAppState({
    dataView
  });
};
const getIsValidColumn = (dataView, esqlQueryColumns) => column => {
  const isValid = esqlQueryColumns ? esqlQueryColumns.some(esqlColumn => esqlColumn.name === column.name) : dataView.fields.getByName(column.name);
  return Boolean(isValid);
};