"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processDocumentCountStats = exports.getDocumentCountStatsRequest = exports.RANDOM_SAMPLER_SEED = void 0;
exports.useDocumentCountStats = useDocumentCountStats;
var _react = require("react");
var _mlStringHash = require("@kbn/ml-string-hash");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _mlIsPopulatedObject = require("@kbn/ml-is-populated-object");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _useObservable = _interopRequireDefault(require("react-use/lib/useObservable"));
var _build_query_filters = require("../../../../common/utils/build_query_filters");
var _kibana_context = require("../../kibana_context");
var _display_error = require("../util/display_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const RANDOM_SAMPLER_SEED = exports.RANDOM_SAMPLER_SEED = 3867418;
const getDocumentCountStatsRequest = (params, randomSamplerWrapper, skipAggs = false) => {
  const {
    index,
    timeFieldName,
    earliest: earliestMs,
    latest: latestMs,
    runtimeFieldMap,
    searchQuery,
    intervalMs,
    fieldsToFetch,
    trackTotalHits
  } = params;
  const size = 0;
  const filterCriteria = (0, _build_query_filters.buildFilterCriteria)(timeFieldName, earliestMs, latestMs, searchQuery);
  const rawAggs = {
    eventRate: {
      date_histogram: {
        field: timeFieldName,
        fixed_interval: `${intervalMs}ms`,
        min_doc_count: 0,
        ...(earliestMs !== undefined && latestMs !== undefined ? {
          extended_bounds: {
            min: earliestMs,
            max: latestMs
          }
        } : {})
      }
    }
  };
  const aggs = randomSamplerWrapper ? randomSamplerWrapper.wrap(rawAggs) : rawAggs;
  const searchBody = {
    query: {
      bool: {
        filter: filterCriteria
      }
    },
    ...(!fieldsToFetch && !skipAggs && timeFieldName !== undefined && intervalMs !== undefined && intervalMs > 0 ? {
      aggs
    } : {}),
    ...((0, _mlIsPopulatedObject.isPopulatedObject)(runtimeFieldMap) ? {
      runtime_mappings: runtimeFieldMap
    } : {}),
    track_total_hits: trackTotalHits === true,
    size
  };
  return {
    index,
    body: searchBody
  };
};
exports.getDocumentCountStatsRequest = getDocumentCountStatsRequest;
const processDocumentCountStats = (body, params, randomSamplerWrapper) => {
  var _ref, _value;
  if (!body) return undefined;
  const totalCount = (_ref = (_value = body.hits.total.value) !== null && _value !== void 0 ? _value : body.hits.total) !== null && _ref !== void 0 ? _ref : 0;
  if (params.intervalMs === undefined || params.earliest === undefined || params.latest === undefined) {
    return {
      totalCount
    };
  }
  const buckets = {};
  const dataByTimeBucket = (0, _lodash.get)(randomSamplerWrapper && body.aggregations !== undefined ? randomSamplerWrapper.unwrap(body.aggregations) : body.aggregations, ['eventRate', 'buckets'], []);
  (0, _lodash.each)(dataByTimeBucket, dataForTime => {
    const time = dataForTime.key;
    buckets[time] = dataForTime.doc_count;
  });
  return {
    interval: params.intervalMs,
    buckets,
    timeRangeEarliest: params.earliest,
    timeRangeLatest: params.latest,
    totalCount
  };
};
exports.processDocumentCountStats = processDocumentCountStats;
function useDocumentCountStats(searchParams, lastRefresh, randomSampler) {
  const {
    data,
    notifications: {
      toasts
    }
  } = (0, _kibana_context.useDataVisualizerKibana)().services;
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const [documentStats, setDocumentStats] = (0, _react.useState)({
    sampleProbability: 1,
    totalCount: 0
  });
  const [documentStatsCache, setDocumentStatsCache] = (0, _react.useState)({});
  const samplingProbability = (0, _useObservable.default)(randomSampler.getProbability$(), randomSampler.getProbability());
  const fetchDocumentCountData = (0, _react.useCallback)(async () => {
    if (!searchParams) return;
    const cacheKey = (0, _mlStringHash.stringHash)(`${JSON.stringify(searchParams)}-${randomSampler.getProbability()}`);
    if (documentStatsCache[cacheKey]) {
      setDocumentStats(documentStatsCache[cacheKey]);
      return;
    }
    try {
      var _totalHitsStats$total;
      abortCtrl.current = new AbortController();
      const totalHitsParams = {
        ...searchParams,
        selectedSignificantTerm: undefined,
        trackTotalHits: true
      };
      const totalHitsResp = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: getDocumentCountStatsRequest(totalHitsParams, undefined, true)
      }, {
        abortSignal: abortCtrl.current.signal
      }));
      const totalHitsStats = processDocumentCountStats(totalHitsResp === null || totalHitsResp === void 0 ? void 0 : totalHitsResp.rawResponse, searchParams);
      const totalCount = (_totalHitsStats$total = totalHitsStats === null || totalHitsStats === void 0 ? void 0 : totalHitsStats.totalCount) !== null && _totalHitsStats$total !== void 0 ? _totalHitsStats$total : 0;
      if (randomSampler) {
        randomSampler.setDocCount(totalCount);
      }
      const randomSamplerWrapper = randomSampler ? randomSampler.createRandomSamplerWrapper() : (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
        totalNumDocs: totalCount,
        seed: RANDOM_SAMPLER_SEED
      });
      const resp = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: getDocumentCountStatsRequest({
          ...searchParams,
          trackTotalHits: false
        }, randomSamplerWrapper)
      }, {
        abortSignal: abortCtrl.current.signal
      }));
      const documentCountStats = processDocumentCountStats(resp === null || resp === void 0 ? void 0 : resp.rawResponse, searchParams, randomSamplerWrapper);
      const newStats = {
        sampleProbability: randomSamplerWrapper.probability,
        documentCountStats,
        totalCount
      };
      setDocumentStatsCache({
        ...documentStatsCache,
        [cacheKey]: newStats
      });
    } catch (error) {
      // An `AbortError` gets triggered when a user cancels a request by navigating away, we need to ignore these errors.
      if (error.name !== 'AbortError') {
        (0, _display_error.displayError)(toasts, searchParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
      }
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data === null || data === void 0 ? void 0 : data.search, documentStatsCache, searchParams, toasts, samplingProbability]);
  (0, _react.useEffect)(function getDocumentCountData() {
    fetchDocumentCountData();
    return () => abortCtrl.current.abort();
  }, [fetchDocumentCountData, lastRefresh, samplingProbability]);

  // Clear the document count stats cache when the outer page (date picker/search bar) triggers a refresh.
  (0, _react.useEffect)(() => {
    setDocumentStatsCache({});
  }, [lastRefresh]);
  return documentStats;
}