"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.discover = discover;
var _rxjs = require("rxjs");
var _repoPackages = require("@kbn/repo-packages");
var _plugin = require("../plugin");
var _plugin_manifest_from_plugin_package = require("./plugin_manifest_from_plugin_package");
var _plugin_context = require("../plugin_context");
var _plugin_manifest_parser = require("./plugin_manifest_parser");
var _scan_plugin_search_paths = require("./scan_plugin_search_paths");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Tries to discover all possible plugins based on the provided plugin config.
 * Discovery result consists of two separate streams, the one (`plugin$`) is
 * for the successfully discovered plugins and the other one (`error$`) is for
 * all the errors that occurred during discovery process.
 *
 * @param config Plugin config instance.
 * @param coreContext Kibana core values.
 * @internal
 */
function discover({
  config,
  coreContext,
  instanceInfo,
  nodeInfo
}) {
  var _coreContext$env$repo;
  const log = coreContext.logger.get('plugins-discovery');
  log.debug('Discovering plugins...');
  if (config.additionalPluginPaths.length && coreContext.env.mode.dev) {
    log.warn(`Explicit plugin paths [${config.additionalPluginPaths}] should only be used in development. Relative imports may not work properly in production.`);
  }
  const fsDiscovery$ = (0, _rxjs.merge)((0, _rxjs.from)(config.additionalPluginPaths), (0, _scan_plugin_search_paths.scanPluginSearchPaths)(config.pluginSearchPaths, log)).pipe((0, _rxjs.concatMap)(pluginPathOrError => {
    return typeof pluginPathOrError === 'string' ? createPlugin$(pluginPathOrError, log, coreContext, instanceInfo, nodeInfo) : [pluginPathOrError];
  }));
  const pluginPkgDiscovery$ = (0, _rxjs.from)((_coreContext$env$repo = coreContext.env.repoPackages) !== null && _coreContext$env$repo !== void 0 ? _coreContext$env$repo : _rxjs.EMPTY).pipe((0, _rxjs.filter)((0, _repoPackages.getPluginPackagesFilter)({
    oss: coreContext.env.cliArgs.oss,
    examples: coreContext.env.cliArgs.runExamples,
    paths: config.additionalPluginPaths,
    parentDirs: config.pluginSearchPaths
  })), (0, _rxjs.map)(pkg => {
    log.debug(`Successfully discovered plugin package "${pkg.id}"`);
    const manifest = (0, _plugin_manifest_from_plugin_package.pluginManifestFromPluginPackage)(coreContext.env.packageInfo.version, pkg.manifest);
    const initializerContext = (0, _plugin_context.createPluginInitializerContext)({
      coreContext,
      opaqueId: Symbol(pkg.id),
      manifest,
      instanceInfo,
      nodeInfo
    });
    return new _plugin.PluginWrapper({
      path: pkg.directory,
      manifest,
      opaqueId: initializerContext.opaqueId,
      initializerContext
    });
  }));
  const discoveryResults$ = (0, _rxjs.merge)(fsDiscovery$, pluginPkgDiscovery$).pipe((0, _rxjs.toArray)(),
  // ensure that everything is always provided in a consistent order
  (0, _rxjs.mergeMap)(pkgs => pkgs.sort((a, b) => {
    const aComp = typeof a !== 'string' ? a.path : a;
    const bComp = typeof b !== 'string' ? b.path : b;
    return aComp.localeCompare(bComp);
  })), (0, _rxjs.shareReplay)());
  return {
    plugin$: discoveryResults$.pipe((0, _rxjs.filter)(entry => entry instanceof _plugin.PluginWrapper)),
    error$: discoveryResults$.pipe((0, _rxjs.filter)(entry => !(entry instanceof _plugin.PluginWrapper)))
  };
}

/**
 * Tries to load and parse the plugin manifest file located at the provided plugin
 * directory path and produces an error result if it fails to do so or plugin manifest
 * isn't valid.
 * @param path Path to the plugin directory where manifest should be loaded from.
 * @param log Plugin discovery logger instance.
 * @param coreContext Kibana core context.
 * @param instanceInfo Info about the instance running Kibana, including uuid.
 * @param nodeRoles Roles this process has been configured with.
 */
function createPlugin$(path, log, coreContext, instanceInfo, nodeInfo) {
  return (0, _rxjs.from)((0, _plugin_manifest_parser.parseManifest)(path, coreContext.env.packageInfo)).pipe((0, _rxjs.map)(manifest => {
    log.debug(`Successfully discovered plugin "${manifest.id}" at "${path}"`);
    const opaqueId = Symbol(manifest.id);
    return new _plugin.PluginWrapper({
      path,
      manifest,
      opaqueId,
      initializerContext: (0, _plugin_context.createPluginInitializerContext)({
        coreContext,
        opaqueId,
        manifest,
        instanceInfo,
        nodeInfo
      })
    });
  }), (0, _rxjs.catchError)(err => [err]));
}