"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDeleteRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerDeleteRoute = (router, {
  config,
  coreUsageData,
  logger,
  access,
  deprecationInfo
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.delete({
    path: '/{type}/{id}',
    options: {
      summary: `Delete a saved object`,
      tags: ['oas-tag:saved objects'],
      access,
      deprecated: deprecationInfo
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the Saved Objects Client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        force: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'delete',
      path: '/api/saved_objects/{type}/{id}',
      request,
      logger
    });
    const {
      type,
      id
    } = request.params;
    const {
      force
    } = request.query;
    const {
      getClient,
      typeRegistry
    } = (await context.core).savedObjects;
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsDelete({
      request,
      types: [type]
    }).catch(() => {});
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfTypeNotVisibleByAPI)(type, typeRegistry);
    }
    const client = getClient();
    const result = await client.delete(type, id, {
      force
    });
    return response.ok({
      body: result
    });
  }));
};
exports.registerDeleteRoute = registerDeleteRoute;