"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCaseFileKinds = exports.createMaxCallback = void 0;
var _constants = require("../../common/constants");
var _types = require("../../common/constants/types");
var _mime_types = require("../../common/constants/mime_types");
var _files = require("../../common/files");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildFileKind = (config, owner, isFipsMode = false) => {
  const hashes = ['sha1', 'sha256'];
  if (!isFipsMode) {
    hashes.unshift('md5');
  }
  return {
    id: (0, _files.constructFileKindIdByOwner)(owner),
    http: fileKindHttpTags(owner),
    maxSizeBytes: createMaxCallback(config),
    allowedMimeTypes: config.allowedMimeTypes,
    hashes
  };
};
const fileKindHttpTags = owner => {
  return {
    create: buildPrivileges(owner, _types.HttpApiPrivilegeOperation.Create),
    download: buildPrivileges(owner, _types.HttpApiPrivilegeOperation.Read),
    getById: buildPrivileges(owner, _types.HttpApiPrivilegeOperation.Read),
    list: buildPrivileges(owner, _types.HttpApiPrivilegeOperation.Read)
  };
};
const buildPrivileges = (owner, operation) => {
  return {
    requiredPrivileges: [(0, _files.constructFilesHttpOperationPrivilege)(owner, operation)]
  };
};
const createMaxCallback = config => file => {
  // if the user set a max size, always return that
  if (config.maxSize != null) {
    return config.maxSize;
  }
  const allowedMimeTypesSet = new Set(config.allowedMimeTypes);

  // if we have the mime type for the file and it exists within the allowed types and it is an image then return the
  // image size
  if (file.mimeType != null && allowedMimeTypesSet.has(file.mimeType) && _mime_types.IMAGE_MIME_TYPES.has(file.mimeType)) {
    return _constants.MAX_IMAGE_FILE_SIZE;
  }
  return _constants.MAX_FILE_SIZE;
};

/**
 * The file kind definition for interacting with the file service for the backend
 */
exports.createMaxCallback = createMaxCallback;
const createFileKinds = (config, isFipsMode = false) => {
  return {
    [_constants.APP_ID]: buildFileKind(config, _constants.APP_ID, isFipsMode),
    [_constants.OBSERVABILITY_OWNER]: buildFileKind(config, _constants.OBSERVABILITY_OWNER, isFipsMode),
    [_constants.SECURITY_SOLUTION_OWNER]: buildFileKind(config, _constants.SECURITY_SOLUTION_OWNER, isFipsMode)
  };
};
const registerCaseFileKinds = (config, filesSetupPlugin, isFipsMode = false) => {
  const fileKinds = createFileKinds(config, isFipsMode);
  for (const fileKind of Object.values(fileKinds)) {
    filesSetupPlugin.registerFileKind(fileKind);
  }
};
exports.registerCaseFileKinds = registerCaseFileKinds;