"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.healthRoute = healthRoute;
var _license_api_access = require("../../lib/license_api_access");
var _track_legacy_route_usage = require("../../lib/track_legacy_route_usage");
var _get_security_health = require("../../lib/get_security_health");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function healthRoute(router, licenseState, encryptedSavedObjects, docLinks, usageCounter, isServerless) {
  router.get({
    path: '/api/alerts/_health',
    validate: false,
    security: _constants.DEFAULT_ALERTING_ROUTE_SECURITY,
    options: {
      access: isServerless ? 'internal' : 'public',
      summary: 'Get the alerting framework health',
      tags: ['oas-tag:alerting'],
      deprecated: {
        documentationUrl: docLinks.links.alerting.legacyRuleApiDeprecations,
        severity: 'warning',
        reason: {
          type: 'migrate',
          newApiMethod: 'GET',
          newApiPath: '/api/alerting/rule/_health'
        }
      }
    }
  }, router.handleLegacyErrors(async function (context, req, res) {
    (0, _license_api_access.verifyApiAccess)(licenseState);
    if (!context.alerting) {
      return res.badRequest({
        body: 'RouteHandlerContext is not registered for alerting'
      });
    }
    (0, _track_legacy_route_usage.trackLegacyRouteUsage)('health', usageCounter);
    try {
      const alertingContext = await context.alerting;
      const rulesClient = await alertingContext.getRulesClient();
      // Verify that user has access to at least one rule type
      const ruleTypes = Array.from(await rulesClient.listRuleTypes());
      if (ruleTypes.length > 0) {
        const alertingFrameworkHealth = await alertingContext.getFrameworkHealth();
        const securityHealth = await (0, _get_security_health.getSecurityHealth)(async () => licenseState ? licenseState.getIsSecurityEnabled() : null, async () => encryptedSavedObjects.canEncrypt, alertingContext.areApiKeysEnabled);
        const frameworkHealth = {
          ...securityHealth,
          alertingFrameworkHealth
        };
        return res.ok({
          body: {
            ...frameworkHealth,
            alertingFrameworkHeath: {
              // Legacy: pre-v8.0 typo
              ...alertingFrameworkHealth,
              _deprecated: 'This state property has a typo, use "alertingFrameworkHealth" instead.'
            }
          }
        });
      } else {
        return res.forbidden({
          body: {
            message: `Unauthorized to access alerting framework health`
          }
        });
      }
    } catch (error) {
      return res.badRequest({
        body: error
      });
    }
  }));
}