"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertIdsOverMaxLimit = getAlertIdsOverMaxLimit;
exports.optimizeTaskStateForFlapping = optimizeTaskStateForFlapping;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function optimizeTaskStateForFlapping(logger, recoveredAlerts = {}, maxAlerts) {
  // this is a space saving effort that will remove the oldest recovered alerts
  // tracked in the task state if the number of alerts we plan to track is over the max alert limit
  const alertIdsOverMaxLimit = getAlertIdsOverMaxLimit(logger, recoveredAlerts, maxAlerts);
  for (const id of alertIdsOverMaxLimit) {
    delete recoveredAlerts[id];
  }
  for (const id of (0, _lodash.keys)(recoveredAlerts)) {
    const alert = recoveredAlerts[id];
    // this is also a space saving effort that will only remove recovered alerts if they are not flapping
    // and if the flapping array does not contain any state changes
    const flapping = alert.getFlapping();
    const flappingHistory = alert.getFlappingHistory() || [];
    const numStateChanges = flappingHistory.filter(f => f).length;
    if (!flapping && numStateChanges === 0) {
      delete recoveredAlerts[id];
    }
  }
  return recoveredAlerts;
}
function getAlertIdsOverMaxLimit(logger, trackedRecoveredAlerts, maxAlerts) {
  const alerts = (0, _lodash.map)(trackedRecoveredAlerts, (alert, id) => {
    return {
      id,
      flappingHistory: alert.getFlappingHistory() || []
    };
  });
  let earlyRecoveredAlertIds = [];
  if (alerts.length > maxAlerts) {
    // alerts are sorted by age using the length of the flapping array
    alerts.sort((a, b) => {
      return a.flappingHistory.length - b.flappingHistory.length;
    });
    earlyRecoveredAlertIds = alerts.slice(maxAlerts).map(alert => alert.id);
    logger.warn(`Recovered alerts have exceeded the max alert limit of ${maxAlerts} : dropping ${earlyRecoveredAlertIds.length} ${earlyRecoveredAlertIds.length > 1 ? 'alerts' : 'alert'}.`);
  }
  return earlyRecoveredAlertIds;
}