"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findRules = findRules;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _filters = require("../../../../rules_client/common/filters");
var _types = require("../../../../authorization/types");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _common = require("../../../../rules_client/common");
var _mapped_params_utils = require("../../../../rules_client/common/mapped_params_utils");
var _constants = require("../../../../rules_client/common/constants");
var _lib = require("../../../../rules_client/lib");
var _saved_objects = require("../../../../saved_objects");
var _schemas = require("./schemas");
var _rule = require("../../../../data/rule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function findRules(context, params) {
  const {
    options,
    excludeFromPublicApi = false,
    includeSnoozeData = false
  } = params || {};
  const {
    fields,
    ruleTypeIds,
    consumers,
    ...restOptions
  } = options || {};
  try {
    if (params) {
      _schemas.findRulesParamsSchema.validate(params);
    }
  } catch (error) {
    throw _boom.default.badRequest(`Error validating find data - ${error.message}`);
  }
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter({
      authorizationEntity: _types.AlertingAuthorizationEntity.Rule,
      filterOpts: _constants.alertingAuthorizationFilterOpts
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.FIND,
      error
    }));
    throw error;
  }
  const {
    filter: authorizationFilter,
    ensureRuleTypeIsAuthorized
  } = authorizationTuple;
  const filterKueryNode = (0, _common.buildKueryNodeFilter)(restOptions.filter);
  let sortField = (0, _common.mapSortField)(restOptions.sortField);
  if (excludeFromPublicApi) {
    try {
      (0, _common.validateOperationOnAttributes)(filterKueryNode, sortField, restOptions.searchFields, context.fieldsToExcludeFromPublicApi);
    } catch (error) {
      throw _boom.default.badRequest(`Error find rules: ${error.message}`);
    }
  }
  sortField = (0, _common.mapSortField)((0, _mapped_params_utils.getModifiedField)(restOptions.sortField));

  // Generate new modified search and search fields, translating certain params properties
  // to mapped_params. Thus, allowing for sort/search/filtering on params.
  // We do the modifcation after the validate check to make sure the public API does not
  // use the mapped_params in their queries.
  const modifiedOptions = {
    ...restOptions,
    ...(restOptions.searchFields && {
      searchFields: (0, _mapped_params_utils.getModifiedSearchFields)(restOptions.searchFields)
    }),
    ...(restOptions.search && {
      search: (0, _mapped_params_utils.getModifiedSearch)(restOptions.searchFields, restOptions.search)
    })
  };

  // Modifies kuery node AST to translate params filter and the filter value to mapped_params.
  // This translation is done in place, and therefore is not a pure function.
  if (filterKueryNode) {
    (0, _mapped_params_utils.modifyFilterKueryNode)({
      astFilter: filterKueryNode
    });
  }
  const ruleTypeIdsFilter = (0, _filters.buildRuleTypeIdsFilter)(ruleTypeIds);
  const consumersFilter = (0, _filters.buildConsumersFilter)(consumers);
  const combinedFilters = (0, _filters.combineFilters)([filterKueryNode, ruleTypeIdsFilter, consumersFilter], 'and');
  const finalFilter = (0, _filters.combineFilterWithAuthorizationFilter)(combinedFilters, authorizationFilter);
  const {
    page,
    per_page: perPage,
    total,
    saved_objects: data
  } = await (0, _rule.findRulesSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    savedObjectsFindOptions: {
      ...modifiedOptions,
      sortField,
      filter: finalFilter,
      fields: fields ? (0, _common.includeFieldsRequiredForAuthentication)(fields) : fields
    }
  });
  const siemRules = [];
  const authorizedData = data.map(({
    id,
    attributes,
    references
  }) => {
    try {
      ensureRuleTypeIsAuthorized(attributes.alertTypeId, attributes.consumer, _types.AlertingAuthorizationEntity.Rule);
    } catch (error) {
      var _context$auditLogger2;
      (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
        action: _audit_events.RuleAuditAction.FIND,
        savedObject: {
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
          id,
          name: attributes.name
        },
        error
      }));
      throw error;
    }
    const rule = (0, _lib.getAlertFromRaw)({
      excludeFromPublicApi,
      id,
      includeLegacyId: false,
      includeSnoozeData,
      isSystemAction: context.isSystemAction,
      logger: context.logger,
      rawRule: fields ? (0, _lodash.pick)(attributes, fields) : attributes,
      references,
      ruleTypeId: attributes.alertTypeId,
      ruleTypeRegistry: context.ruleTypeRegistry
    });

    // collect SIEM rule for further formatting legacy actions
    if (attributes.consumer === _ruleDataUtils.AlertConsumers.SIEM) {
      siemRules.push(rule);
    }
    return rule;
  });
  authorizedData.forEach(({
    id,
    name
  }) => {
    var _context$auditLogger3;
    return (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.FIND,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id,
        name
      }
    }));
  });

  // format legacy actions for SIEM rules, if there any
  if (siemRules.length) {
    const formattedRules = await (0, _lib.formatLegacyActions)(siemRules, {
      savedObjectsClient: context.unsecuredSavedObjectsClient,
      logger: context.logger
    });
    const formattedRulesMap = formattedRules.reduce((acc, rule) => {
      acc[rule.id] = rule;
      return acc;
    }, {});
    return {
      page,
      perPage,
      total,
      // replace siem formatted rules
      data: authorizedData.map(rule => {
        var _formattedRulesMap$ru;
        return (_formattedRulesMap$ru = formattedRulesMap[rule.id]) !== null && _formattedRulesMap$ru !== void 0 ? _formattedRulesMap$ru : rule;
      })
    };
  }
  return {
    page,
    perPage,
    total,
    data: authorizedData
  };
}