"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteRule = deleteRule;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _authorization = require("../../../../authorization");
var _retry_if_conflicts = require("../../../../lib/retry_if_conflicts");
var _bulk_mark_api_keys_for_invalidation = require("../../../../invalidate_pending_api_keys/bulk_mark_api_keys_for_invalidation");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _lib = require("../../../../rules_client/lib");
var _saved_objects = require("../../../../saved_objects");
var _schemas = require("./schemas");
var _rule = require("../../../../data/rule");
var _soft_delete_gaps = require("../../../../lib/rule_gaps/soft_delete/soft_delete_gaps");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function deleteRule(context, params) {
  try {
    _schemas.deleteRuleParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating delete params - ${error.message}`);
  }
  const {
    id
  } = params;
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `rulesClient.delete('${id}')`, async () => await deleteRuleWithOCC(context, {
    id
  }));
}
async function deleteRuleWithOCC(context, {
  id
}) {
  var _context$auditLogger2;
  let taskIdToRemove;
  let apiKeyToInvalidate = null;
  let apiKeyCreatedByUser = false;
  let attributes;
  let rule;
  try {
    const decryptedRule = await (0, _rule.getDecryptedRuleSo)({
      encryptedSavedObjectsClient: context.encryptedSavedObjectsClient,
      id,
      savedObjectsGetOptions: {
        namespace: context.namespace
      }
    });
    apiKeyToInvalidate = decryptedRule.attributes.apiKey;
    apiKeyCreatedByUser = decryptedRule.attributes.apiKeyCreatedByUser;
    taskIdToRemove = decryptedRule.attributes.scheduledTaskId;
    attributes = decryptedRule.attributes;
    rule = decryptedRule;
  } catch (e) {
    // We'll skip invalidating the API key since we failed to load the decrypted saved object
    context.logger.error(`delete(): Failed to load API key to invalidate on alert ${id}: ${e.message}`);

    // Still attempt to load the scheduledTaskId using SOC
    rule = await (0, _rule.getRuleSo)({
      savedObjectsClient: context.unsecuredSavedObjectsClient,
      id
    });
    taskIdToRemove = rule.attributes.scheduledTaskId;
    attributes = rule.attributes;
  }
  try {
    await context.authorization.ensureAuthorized({
      ruleTypeId: attributes.alertTypeId,
      consumer: attributes.consumer,
      operation: _authorization.WriteOperations.Delete,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.DELETE,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id,
        name: attributes.name
      },
      error
    }));
    throw error;
  }
  await (0, _lib.untrackRuleAlerts)(context, id, attributes);

  // migrate legacy actions only for SIEM rules
  // TODO (http-versioning): Remove this cast, this enables us to move forward
  // without fixing all of other solution types
  if (attributes.consumer === _ruleDataUtils.AlertConsumers.SIEM) {
    await (0, _lib.bulkMigrateLegacyActions)({
      context,
      rules: [rule],
      skipActionsValidation: true
    });
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.DELETE,
    outcome: 'unknown',
    savedObject: {
      type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
      id,
      name: attributes.name
    }
  }));
  try {
    const eventLogClient = await context.getEventLogClient();
    await (0, _soft_delete_gaps.softDeleteGaps)({
      ruleId: id,
      logger: context.logger,
      eventLogClient,
      eventLogger: context.eventLogger
    });
  } catch (error) {
    // Failing to soft delete gaps should not block the rule deletion
    context.logger.error(`delete(): Failed to soft delete gaps for rule ${id}: ${error.message}`);
  }
  const removeResult = await (0, _rule.deleteRuleSo)({
    savedObjectsClient: context.unsecuredSavedObjectsClient,
    id
  });
  await Promise.all([taskIdToRemove ? context.taskManager.removeIfExists(taskIdToRemove) : null, context.backfillClient.deleteBackfillForRules({
    ruleIds: [id],
    namespace: context.namespace,
    unsecuredSavedObjectsClient: context.unsecuredSavedObjectsClient
  }), apiKeyToInvalidate && !apiKeyCreatedByUser ? (0, _bulk_mark_api_keys_for_invalidation.bulkMarkApiKeysForInvalidation)({
    apiKeys: [apiKeyToInvalidate]
  }, context.logger, context.unsecuredSavedObjectsClient) : null]);
  return removeResult;
}