"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.routeHandlerFactory = routeHandlerFactory;
var _apmUtils = require("@kbn/apm-utils");
var _mlRouteUtils = require("@kbn/ml-route-utils");
var _constants = require("@kbn/aiops-common/constants");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _track_route_usage = require("../../lib/track_route_usage");
var _response_stream_factory = require("./response_stream_factory");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The log rate analysis route handler sets up `responseStreamFactory`
 * to create the response stream and then uses its handlers to
 * walk through the steps of the analysis.
 */
function routeHandlerFactory(version, license, logger, coreStart, usageCounter) {
  return async (context, request, response) => {
    const {
      headers
    } = request;
    (0, _track_route_usage.trackAIOpsRouteUsage)(`POST ${_constants.AIOPS_API_ENDPOINT.LOG_RATE_ANALYSIS}`, headers[_constants.AIOPS_ANALYSIS_RUN_ORIGIN], usageCounter);
    if (!license.isActivePlatinumLicense) {
      return response.forbidden();
    }
    const esClient = (await context.core).elasticsearch.client.asCurrentUser;
    const executionContext = (0, _mlRouteUtils.createExecutionContext)(coreStart, _constants.AIOPS_PLUGIN_ID, request.route.path);
    return await coreStart.executionContext.withContext(executionContext, () => {
      const {
        analysis,
        logDebugMessage,
        stateHandler,
        responseStream,
        responseWithHeaders
      } = (0, _response_stream_factory.responseStreamFactory)({
        version,
        esClient,
        requestBody: request.body,
        events: request.events,
        headers: request.headers,
        logger
      });
      async function runAnalysis() {
        try {
          logDebugMessage('Starting analysis.');
          logDebugMessage(`Sample probability: ${stateHandler.sampleProbability()}`);
          stateHandler.isRunning(true);
          analysis.overridesHandler();
          responseStream.pushPingWithTimeout();

          // Step 1: Index Info: Field candidates and zero docs fallback flag
          const indexInfo = await (0, _apmUtils.withSpan)({
            name: 'fetch_index_info',
            type: 'aiops-log-rate-analysis'
          }, () => analysis.indexInfoHandler());
          if (!indexInfo) {
            return;
          }

          // Step 2: Significant categories and terms
          const significantItemsObj = indexInfo.zeroDocsFallback ? await (0, _apmUtils.withSpan)({
            name: 'fetch_top_items',
            type: 'aiops-log-rate-analysis'
          }, () => analysis.topItemsHandler(indexInfo)) : await (0, _apmUtils.withSpan)({
            name: 'fetch_significant_items',
            type: 'aiops-log-rate-analysis'
          }, () => analysis.significantItemsHandler(indexInfo));
          if (!significantItemsObj) {
            return;
          }
          const {
            fieldValuePairsCount,
            significantCategories,
            significantTerms
          } = significantItemsObj;

          // Step 3: Fetch overall histogram
          const overallTimeSeries = await (0, _apmUtils.withSpan)({
            name: 'fetch_overall_timeseries',
            type: 'aiops-log-rate-analysis'
          }, () => analysis.overallHistogramHandler());

          // Step 4: Histograms
          await (0, _apmUtils.withSpan)({
            name: 'significant-item-histograms',
            type: 'aiops-log-rate-analysis'
          }, () => analysis.histogramHandler(fieldValuePairsCount, significantCategories, significantTerms, overallTimeSeries));

          // Step 5: Smart grouping
          if (stateHandler.groupingEnabled()) {
            await (0, _apmUtils.withSpan)({
              name: 'grouping-with-histograms',
              type: 'aiops-log-rate-analysis'
            }, () => analysis.groupingHandler(significantCategories, significantTerms, overallTimeSeries));
          }
          responseStream.endWithUpdatedLoadingState();
        } catch (e) {
          if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
            logger.error(`Log Rate Analysis failed to finish, got: \n${e.toString()}`);
            responseStream.pushError(`Log Rate Analysis failed to finish.`);
          }
          responseStream.end();
        }
      }

      // Do not call this using `await` so it will run asynchronously while we return the stream already.
      void runAnalysis();
      return response.ok(responseWithHeaders);
    });
  };
}