"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignificantTermRequest = exports.fetchSignificantTermPValues = void 0;
var _lodash = require("lodash");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _constants = require("../constants");
var _get_normalized_score = require("./get_normalized_score");
var _get_query_with_params = require("./get_query_with_params");
var _get_request_base = require("./get_request_base");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO Consolidate with duplicate `fetchDurationFieldCandidates` in
// `x-pack/solutions/observability/plugins/apm/server/routes/correlations/queries/fetch_failed_events_correlation_p_values.ts`

const getSignificantTermRequest = (params, fieldNames, {
  wrap
}) => {
  var _params$timeFieldName;
  const query = (0, _get_query_with_params.getQueryWithParams)({
    params
  });
  const timeFieldName = (_params$timeFieldName = params.timeFieldName) !== null && _params$timeFieldName !== void 0 ? _params$timeFieldName : '@timestamp';
  let filter = [];
  if (query.bool && Array.isArray(query.bool.filter)) {
    filter = query.bool.filter.filter(d => Object.keys(d)[0] !== 'range');
    query.bool.filter = [...filter, {
      range: {
        [timeFieldName]: {
          gte: params.deviationMin,
          lt: params.deviationMax,
          format: 'epoch_millis'
        }
      }
    }];
  }
  const fieldCandidateAggs = fieldNames.reduce((aggs, fieldName, index) => {
    // Used to identify fields with only one distinct value which we'll ignore in the analysis.
    aggs[`distinct_count_${index}`] = {
      cardinality: {
        field: fieldName
      }
    };

    // Used to calculate the p-value for the field.
    aggs[`sig_term_p_value_${index}`] = {
      significant_terms: {
        field: fieldName,
        background_filter: {
          bool: {
            filter: [...filter, {
              range: {
                [timeFieldName]: {
                  gte: params.baselineMin,
                  lt: params.baselineMax,
                  format: 'epoch_millis'
                }
              }
            }]
          }
        },
        // @ts-expect-error `p_value` is not yet part of `AggregationsAggregationContainer`
        p_value: {
          background_is_superset: false
        },
        size: 1000
      }
    };
    return aggs;
  }, {});
  const body = {
    query,
    size: 0,
    aggs: wrap(fieldCandidateAggs)
  };
  return {
    ...(0, _get_request_base.getRequestBase)(params),
    body
  };
};
exports.getSignificantTermRequest = getSignificantTermRequest;
const fetchSignificantTermPValues = async ({
  esClient,
  abortSignal,
  logger,
  emitError,
  arguments: args
}) => {
  // The default value of 1 means no sampling will be used
  const {
    fieldNames,
    sampleProbability = 1,
    ...params
  } = args;
  const randomSamplerWrapper = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const result = [];
  const resp = await esClient.search(getSignificantTermRequest(params, fieldNames, randomSamplerWrapper), {
    signal: abortSignal,
    maxRetries: 0
  });
  if (resp.aggregations === undefined) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(resp)) {
      if (logger) {
        logger.error(`Failed to fetch p-value aggregation for field names ${fieldNames.join()}, got: \n${JSON.stringify(resp, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch p-value aggregation for field names "${fieldNames.join()}".`);
      }
    }
    return result;
  }
  const unwrappedResp = randomSamplerWrapper.unwrap(resp.aggregations);
  for (const [index, fieldName] of fieldNames.entries()) {
    const pValueBuckets = unwrappedResp[`sig_term_p_value_${index}`];
    const distinctCount = unwrappedResp[`distinct_count_${index}`].value;
    for (const bucket of pValueBuckets.buckets) {
      const pValue = Math.exp(-bucket.score);
      if (typeof pValue === 'number' &&
      // Skip items where the p-value is not significant.
      pValue < _constants.LOG_RATE_ANALYSIS_SETTINGS.P_VALUE_THRESHOLD &&
      // Skip items where the field has only one distinct value.
      distinctCount > 1) {
        result.push({
          key: `${fieldName}:${String(bucket.key)}`,
          type: _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD,
          fieldName,
          fieldValue: String(bucket.key),
          doc_count: bucket.doc_count,
          bg_count: bucket.bg_count,
          total_doc_count: pValueBuckets.doc_count,
          total_bg_count: pValueBuckets.bg_count,
          score: bucket.score,
          pValue,
          normalizedScore: (0, _get_normalized_score.getNormalizedScore)(bucket.score)
        });
      }
    }
  }
  return (0, _lodash.uniqBy)(result, d => `${d.fieldName},${d.fieldValue}`);
};
exports.fetchSignificantTermPValues = fetchSignificantTermPValues;