"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextService = void 0;
var tslib_1 = require("tslib");
var rxjs_1 = require("rxjs");
var validation_1 = require("../schema/validation");
var ContextService = /** @class */ (function () {
    function ContextService(context$, isDevMode, logger) {
        this.context$ = context$;
        this.isDevMode = isDevMode;
        this.logger = logger;
        this.contextProvidersRegistry = new Map();
        this.contextProvidersSubscriptions = new Map();
    }
    /**
     * Registers a context provider, and subscribes to any updates from it.
     * @param contextProviderOpts The options to register the context provider {@link ContextProviderOpts}
     */
    ContextService.prototype.registerContextProvider = function (_a) {
        var _this = this;
        var name = _a.name, context$ = _a.context$, schema = _a.schema;
        if (this.contextProvidersSubscriptions.has(name)) {
            throw new Error("Context provider with name '".concat(name, "' already registered"));
        }
        // Declare the validator only in dev-mode
        var validator = this.isDevMode ? (0, validation_1.schemaToIoTs)(schema) : undefined;
        var subscription = context$
            .pipe((0, rxjs_1.filter)(function (context) {
            if (validator) {
                try {
                    (0, validation_1.validateSchema)("Context Provider '".concat(name, "'"), validator, context);
                }
                catch (validationError) {
                    _this.logger.error(validationError);
                    return false;
                }
            }
            return true;
        }))
            .subscribe(function (context) {
            // We store each context linked to the context provider, so they can increase and reduce
            // the number of fields they report without having left-overs in the global context.
            _this.contextProvidersRegistry.set(name, context);
            // For every context change, we rebuild the global context.
            // It's better to do it here than to rebuild it for every reportEvent.
            _this.updateGlobalContext();
        });
        this.contextProvidersSubscriptions.set(name, subscription);
    };
    /**
     * Removes the context provider from the registry, unsubscribes from it, and rebuilds the global context.
     * @param name The name of the context provider to remove.
     */
    ContextService.prototype.removeContextProvider = function (name) {
        var _a;
        (_a = this.contextProvidersSubscriptions.get(name)) === null || _a === void 0 ? void 0 : _a.unsubscribe();
        this.contextProvidersRegistry.delete(name);
        this.updateGlobalContext();
    };
    /**
     * Loops through all the context providers and sets the global context
     * @private
     */
    ContextService.prototype.updateGlobalContext = function () {
        var _this = this;
        this.context$.next(tslib_1.__spreadArray([], tslib_1.__read(this.contextProvidersRegistry.values()), false).reduce(function (acc, context) {
            return tslib_1.__assign(tslib_1.__assign({}, acc), _this.removeEmptyValues(context));
        }, {}));
    };
    ContextService.prototype.removeEmptyValues = function (context) {
        if (!isObject(context)) {
            return {};
        }
        return Object.keys(context).reduce(function (acc, key) {
            if (context[key] !== undefined) {
                acc[key] = context[key];
            }
            return acc;
        }, {});
    };
    return ContextService;
}());
exports.ContextService = ContextService;
function isObject(value) {
    return typeof value === 'object' && value !== null;
}
