"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isStepReadyToComplete = exports.isStepInProgress = exports.isLastStep = exports.isGuideActive = exports.getStepConfig = exports.getInProgressStepId = exports.getInProgressStepConfig = exports.getCompletedSteps = exports.findGuideConfigByGuideId = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const findGuideConfigByGuideId = (guidesConfig, guideId) => {
  if (guidesConfig && guideId && Object.keys(guidesConfig).includes(guideId)) {
    return guidesConfig[guideId];
  }
};
exports.findGuideConfigByGuideId = findGuideConfigByGuideId;
const getStepConfig = (guideConfig, guideId, stepId) => {
  return guideConfig === null || guideConfig === void 0 ? void 0 : guideConfig.steps.find(step => step.id === stepId);
};
exports.getStepConfig = getStepConfig;
const getStepIndex = (guideConfig, guideId, stepId) => {
  if (guideConfig) {
    return guideConfig.steps.findIndex(step => step.id === stepId);
  }
  return -1;
};
const isLastStep = (guideConfig, guideId, stepId) => {
  const activeStepIndex = getStepIndex(guideConfig, guideId, stepId);
  const stepsNumber = (guideConfig === null || guideConfig === void 0 ? void 0 : guideConfig.steps.length) || 0;
  if (stepsNumber > 0) {
    return activeStepIndex === stepsNumber - 1;
  }
  return false;
};
exports.isLastStep = isLastStep;
const getInProgressStepId = state => {
  const inProgressStep = state.steps.find(step => step.status === 'in_progress');
  return inProgressStep ? inProgressStep.id : undefined;
};
exports.getInProgressStepId = getInProgressStepId;
const getInProgressStepConfig = (guideConfig, state) => {
  const inProgressStepId = getInProgressStepId(state);
  if (inProgressStepId) {
    if (guideConfig) {
      return guideConfig.steps.find(step => step.id === inProgressStepId);
    }
  }
};
exports.getInProgressStepConfig = getInProgressStepConfig;
const isGuideActive = (pluginState, guideId) => {
  // false if pluginState is undefined or plugin state is not in progress
  // or active guide is undefined
  if (!pluginState || pluginState.status !== 'in_progress' || !pluginState.activeGuide) {
    return false;
  }
  // guideId is passed, check that it's the id of the active guide
  if (guideId) {
    const {
      activeGuide
    } = pluginState;
    return !!(activeGuide.isActive && activeGuide.guideId === guideId);
  }
  return true;
};
exports.isGuideActive = isGuideActive;
const isStepStatus = (guideState, status, guideId, stepId) => {
  if (!guideState || !guideState.isActive || guideState.guideId !== guideId) return false;

  // false if the step is not 'in_progress'
  const selectedStep = guideState.steps.find(step => step.id === stepId);
  return selectedStep ? selectedStep.status === status : false;
};
const isStepInProgress = (guideState, guideId, stepId) => {
  return isStepStatus(guideState, 'in_progress', guideId, stepId);
};
exports.isStepInProgress = isStepInProgress;
const isStepReadyToComplete = (guideState, guideId, stepId) => {
  return isStepStatus(guideState, 'ready_to_complete', guideId, stepId);
};
exports.isStepReadyToComplete = isStepReadyToComplete;
const getCompletedSteps = (guideState, stepId, setToReadyToComplete) => {
  const currentStepIndex = guideState.steps.findIndex(step => step.id === stepId);
  const currentStep = guideState.steps[currentStepIndex];
  return guideState.steps.map((step, stepIndex) => {
    const isCurrentStep = step.id === currentStep.id;
    const isNextStep = stepIndex === currentStepIndex + 1;
    if (isCurrentStep) {
      return {
        id: step.id,
        status: setToReadyToComplete ? 'ready_to_complete' : 'complete'
      };
    }

    // if the current step is being updated to 'ready_to_complete, the next step stays inactive
    // otherwise update the next step to active status
    if (isNextStep) {
      return setToReadyToComplete ? step : {
        id: step.id,
        status: 'active'
      };
    }

    // All other steps return as-is
    return step;
  });
};
exports.getCompletedSteps = getCompletedSteps;